/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static java.util.Collections.emptySet;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

import org.apache.maven.doxia.sink.Sink;
import org.apache.maven.doxia.sink.SinkFactory;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Execute;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.reporting.MavenMultiPageReport;
import org.apache.maven.reporting.MavenReport;
import org.apache.maven.reporting.MavenReportException;
import org.mule.munit.coverage.printer.ConsolePrinter;
import org.mule.munit.coverage.printer.HtmlPrinter;
import org.mule.munit.coverage.printer.JsonPrinter;
import org.mule.munit.coverage.printer.MavenCoverageReportPrinter;
import org.mule.munit.remote.coverage.report.model.ApplicationCoverageReport;

import com.google.gson.Gson;

/**
 * A mojo to generate reports regarding the MUnit test run
 * 
 * @author Mulesoft Inc.
 */
@Mojo(name = "coverage-report", defaultPhase = LifecyclePhase.TEST)
@Execute(lifecycle = "munit", phase = LifecyclePhase.TEST)
public class MUnitCoverageReportMojo extends AbstractMojo implements MavenMultiPageReport {

  private static final String WONT_GENERATE_REPORT_MESSAGE = "This report cannot be generated as part of the current build";

  private static final String MUNIT_FOLDER = "munit";
  private static final String COVERAGE_FOLDER = "coverage";

  /**
   * Output directory for the reports. Note that this parameter is only relevant if the goal is run from the command line or from
   * the default build lifecycle. If the goal is run indirectly as part of a site generation, the output directory configured in
   * the Maven Site Plugin is used instead.
   */
  @Parameter(defaultValue = "${project.reporting.outputDirectory}/munit/coverage/")
  protected File outputDirectory;

  @Parameter(property = "munit.coverageConfigData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-config.data")
  protected File coverageConfigDataFile;

  @Parameter(property = "munit.coverageReportData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-report.data")
  protected File coverageReportDataFile;

  @Override
  public boolean canGenerateReport() {
    return coverageConfigDataFile.exists() && coverageReportDataFile.exists();
  }

  @Override
  public void execute() throws MojoExecutionException {
    if (!canGenerateReport()) {
      return;
    }
    try {
      doGenerateReport(Locale.getDefault(), readCoverageReportData(), readCoverageConfigurationData());
    } catch (MavenReportException e) {
      throw new MojoExecutionException("An error has occurred in " + getName(Locale.getDefault()) + " report generation.", e);
    }
  }

  @Override
  public boolean isExternalReport() {
    try {
      return hasHtmlFormat(readCoverageConfigurationData());
    } catch (MavenReportException e) {
      return false;
    }
  }

  /**
   * Generates the coverage report when executing the <i>site</i> lifecycle
   */
  @Override
  public void generate(Sink sink, SinkFactory sinkFactory, Locale locale) throws MavenReportException {
    if (!canGenerateReport()) {
      getLog().info(WONT_GENERATE_REPORT_MESSAGE + ": Coverage data is missing");
      return;
    }
    Coverage coverageConfig = readCoverageConfigurationData();
    if (!hasHtmlFormat(coverageConfig)) {
      getLog().warn(WONT_GENERATE_REPORT_MESSAGE + ": Html format is not present in the plugin configuration");
      generateWarningHtml(sink);
      return;
    }
    doGenerateReport(locale, readCoverageReportData(), coverageConfig);
    sink.close();
  }

  @Override
  public void generate(org.codehaus.doxia.sink.Sink sink, Locale locale) throws MavenReportException {
    doGenerateReport(locale, readCoverageReportData(), readCoverageConfigurationData());
    sink.close();
  }

  @Override
  public String getOutputName() {
    return "munit/coverage/summary";
  }

  @Override
  public String getCategoryName() {
    return MavenReport.CATEGORY_PROJECT_REPORTS;
  }

  @Override
  public String getName(Locale locale) {
    return "MUnit Coverage";
  }

  @Override
  public String getDescription(Locale locale) {
    return "MUnit - Coverage Report";
  }

  @Override
  public void setReportOutputDirectory(File reportOutputDirectory) {
    if (reportOutputDirectory != null && (!reportOutputDirectory.getParentFile().getName().equals(MUNIT_FOLDER)
        || !reportOutputDirectory.getName().equals(COVERAGE_FOLDER))) {
      outputDirectory = reportOutputDirectory.toPath().resolve(MUNIT_FOLDER).resolve(COVERAGE_FOLDER).toFile();
    } else {
      outputDirectory = reportOutputDirectory;
    }
  }

  @Override
  public File getReportOutputDirectory() {
    return outputDirectory;
  }

  protected Set<MavenCoverageReportPrinter> buildPrinters(Coverage coverageConfiguration) {
    if (coverageConfiguration == null || coverageConfiguration.getFormats() == null) {
      return emptySet();
    }
    Set<MavenCoverageReportPrinter> printers = new HashSet<>();
    outputDirectory.mkdirs();
    for (String printFormat : coverageConfiguration.getFormats()) {
      if (ConsolePrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
        printers.add(new ConsolePrinter(getLog()));
        continue;
      }

      if (HtmlPrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
        printers.add(new HtmlPrinter(coverageConfiguration, outputDirectory));
        continue;
      }

      if (JsonPrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
        printers.add(new JsonPrinter(coverageConfiguration, new File(outputDirectory, "munit-coverage.json")));
        continue;
      }

      getLog().debug("No report format find for: " + printFormat);
    }
    return printers;
  }

  protected void doGenerateReport(Locale locale, ApplicationCoverageReport coverageReport, Coverage coverageConfiguration)
      throws MavenReportException {
    if (coverageReport == null) {
      getLog().warn(WONT_GENERATE_REPORT_MESSAGE + ": Coverage data not found");
      return;
    }
    Set<MavenCoverageReportPrinter> printers = buildPrinters(coverageConfiguration);
    printers.forEach(printer -> {
      getLog().info("Generating: " + printer.getPrinterName() + " report...");
      printer.printReport(coverageReport);
    });
  }

  protected Coverage readCoverageConfigurationData() throws MavenReportException {
    return readFileToClass(coverageConfigDataFile, Coverage.class);
  }

  protected ApplicationCoverageReport readCoverageReportData() throws MavenReportException {
    return readFileToClass(coverageReportDataFile, ApplicationCoverageReport.class);
  }

  private void generateWarningHtml(Sink sink) {
    sink.text("To enable the Html Coverage Report, add the html format to the plugin configuration");
    sink.close();
  }

  private <T> T readFileToClass(File dataFile, Class<T> clazz) throws MavenReportException {
    try {
      FileReader fileReader = new FileReader(dataFile);
      T coverageReport = new Gson().fromJson(fileReader, clazz);
      fileReader.close();
      return coverageReport;
    } catch (IOException e) {
      throw new MavenReportException("An error occurred while reading the report data", e);
    }
  }

  private boolean hasHtmlFormat(Coverage coverageConfig) {
    return coverageConfig != null && coverageConfig.getFormats() != null
        && coverageConfig.getFormats().stream().anyMatch(HtmlPrinter.PRINTER_NAME::equalsIgnoreCase);
  }

}
