/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static org.mule.munit.mojo.RuntimeProducts.EE;
import static org.mule.munit.remote.FolderNames.MUNIT_WORKING_DIR;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.toolchain.ToolchainManager;

import org.mule.munit.common.util.FileUtils;
import org.mule.munit.coverage.CoverageLimitsChecker;
import org.mule.munit.mojo.JVMLocator;
import org.mule.munit.mojo.MessageHandlerFactory;
import org.mule.munit.mojo.ResultPrinterFactory;
import org.mule.munit.mojo.exceptions.MojoExecutionExceptionFactory;
import org.mule.munit.remote.ApplicationStructureGenerator;
import org.mule.munit.remote.RemoteRunner;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.remote.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.runner.JVMStarter;
import org.mule.munit.runner.consumer.ErrorStreamConsumer;
import org.mule.munit.runner.consumer.RunnerStreamConsumer;
import org.mule.munit.runner.model.RunResult;
import org.mule.munit.runner.printer.ResultPrinter;
import org.mule.munit.util.ArgLinesManager;
import org.mule.munit.util.ClasspathManager;
import org.mule.munit.util.JarFileFactory;
import org.mule.munit.util.RunConfigurationFactory;
import org.mule.munit.util.properties.UserPropertiesBuilder;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;


/**
 * <p>
 * MUnit Mojo to run tests
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
@Mojo(name = "test",
    defaultPhase = LifecyclePhase.TEST,
    requiresDependencyResolution = ResolutionScope.TEST)
public class MUnitMojo extends AbstractMojo {

  private static final String SKIP_TESTS_PROPERTY = "skipTests";
  private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";

  private static final String ARG_TOKEN = "-";
  public static final String STARTER_CLASS_FILE = "munitstarter";
  private static final Class REMOTE_RUNNER_CLASS = RemoteRunner.class;
  private static final String RUN_CONFIGURATION_ARG = ARG_TOKEN + RunConfigurationParser.RUN_CONFIGURATION_PARAMETER;
  public static final String RUN_CONFIGURATION_JSON = "run-configuration.json";

  public static final String MULE_ARTIFACT_JSON_FILE_NAME = "mule-artifact.json";
  public static final String MUNIT_PREVIOUS_RUN_PLACEHOLDER = "MUNIT_PREVIOUS_RUN_PLACEHOLDER";


  @Parameter(property = "project", required = true)
  protected MavenProject project;

  @Parameter(property = "project.testClasspathElements", required = true, readonly = true)
  protected List<String> classpathElements;

  @Parameter(property = "munit.test")
  protected String munitTest;

  @Parameter(property = "munit.tags")
  protected String munitTags;

  @Parameter(defaultValue = "${skipMunitTests}")
  protected boolean skipMunitTests = false;

  @Parameter(property = "skipAfterFailure", defaultValue = "false")
  protected boolean skipAfterFailure = false;

  @Parameter(property = "system.property.variables")
  protected Map<String, String> systemPropertyVariables;

  @Parameter(property = "environment.variables")
  protected Map<String, String> environmentVariables;

  @Parameter(property = "dynamic.ports")
  protected List<String> dynamicPorts;

  @Parameter(property = "munit.coverage")
  protected Coverage coverage;

  @Parameter(defaultValue = "${munit.randomFailMessages}")
  private boolean randomFailMessages = false;

  @Parameter(property = "argLines")
  protected List<String> argLines;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit")
  protected File munitTestsDirectory;

  @Parameter(property = "redirectTestOutputToFile", defaultValue = "false")
  protected boolean redirectTestOutputToFile;

  @Parameter(defaultValue = "${project.build.directory}/munit-reports/output/")
  protected File testOutputDirectory;

  @Parameter(property = "enableSurefireReports", defaultValue = "true")
  protected boolean enableSurefireReports;

  @Parameter(defaultValue = "${project.build.directory}/surefire-reports/")
  protected File surefireReportsFolder;

  @Parameter
  protected String runtimeVersion;

  @Parameter(property = "runtimeProduct", defaultValue = "MULE_EE")
  protected String runtimeProduct;

  @Parameter(defaultValue = "${session}", readonly = true)
  protected MavenSession session;

  @Parameter(defaultValue = "${plugin.version}")
  protected String pluginVersion;

  @Parameter(property = "munit.coverageReportData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-report.data")
  protected File coverageReportDataFile;

  @Parameter(property = "munit.coverageConfigData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-config.data")
  protected File coverageConfigDataFile;

  @Component
  protected ToolchainManager toolchainManager;

  private File munitWorkingDirectory;
  private Gson gson = new GsonBuilder().setPrettyPrinting().create();

  protected MojoExecutionExceptionFactory exceptionFactory;

  protected ClasspathManager classpathManager;
  protected CoverageLimitsChecker coverageLimitsChecker;
  protected ResultPrinterFactory resultPrinterFactory;
  protected MessageHandlerFactory messageHandlerFactory;
  protected RunConfigurationFactory runConfigurationFactory;
  protected JVMLocator jvmLocator;
  protected JarFileFactory jarFileFactory;
  protected Map<String, String> effectiveSystemProperties;


  public void execute() throws MojoExecutionException {
    if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {
      if (!skipMunitTests) {
        if (!hasExecutedBefore()) {
          init();
          doExecute();
        } else {
          getLog().info("Skipping execution of munit because it has already been run");
        }
      } else {
        getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_MUNIT_TESTS_PROPERTY + "] was set to true");
      }
    } else {
      getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_TESTS_PROPERTY + "] was set to true");
    }
  }

  /**
   * This method avoids running MUnit more than once. This could happen because other phases such as the coverage-report, can
   * trigger the "test" phase since it needs its information.
   * 
   * @return true if the MUnit run has already happened before
   */
  @SuppressWarnings("unchecked")
  protected boolean hasExecutedBefore() {
    Map<String, String> pluginContext = getPluginContext();
    if (pluginContext.containsKey(MUNIT_PREVIOUS_RUN_PLACEHOLDER)) {
      return true;
    }
    getPluginContext().put(MUNIT_PREVIOUS_RUN_PLACEHOLDER, MUNIT_PREVIOUS_RUN_PLACEHOLDER);
    return false;
  }

  protected void init() throws MojoExecutionException {
    this.exceptionFactory = new MojoExecutionExceptionFactory(randomFailMessages);
    this.messageHandlerFactory = new MessageHandlerFactory(getLog());
    this.effectiveSystemProperties = getEffectiveSystemProperties();

    this.resultPrinterFactory = new ResultPrinterFactory(getLog())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory)
        .withCoverageSummaryReport(coverage, runtimeProduct);

    this.munitWorkingDirectory = createAndGetMunitWorkingDirectory();
    this.runConfigurationFactory = new RunConfigurationFactory(getLog(),
                                                               munitTest, munitTags, skipAfterFailure,
                                                               runtimeVersion, runtimeProduct,
                                                               munitWorkingDirectory,
                                                               coverage, pluginVersion, project, session);


    this.jvmLocator = new JVMLocator(session, toolchainManager, getLog());
    this.classpathManager = new ClasspathManager(classpathElements, REMOTE_RUNNER_CLASS, getLog());
    this.jarFileFactory = new JarFileFactory();

    this.coverageLimitsChecker = new CoverageLimitsChecker(coverage, runtimeProduct, getLog());
  }

  protected void doExecute() throws MojoExecutionException {
    generateApplicationStructure();

    RunConfiguration runConfiguration = runConfigurationFactory.createRunConfiguration();
    try {

      if (runConfiguration == null) {
        getLog().info("No Run configuration created not running MUnit ");
        return;
      }
      if (runConfiguration.getSuitePaths().isEmpty()) {
        getLog().info("No MUnit suite files were found to run");
        return;
      }

      JVMStarter jvmStarter = createJVMStarter(runConfiguration);
      RunnerStreamConsumer streamConsumer = new RunnerStreamConsumer(messageHandlerFactory.create(redirectTestOutputToFile));
      ErrorStreamConsumer errorStreamConsumer = new ErrorStreamConsumer(redirectTestOutputToFile);
      int result = jvmStarter.execute(streamConsumer, errorStreamConsumer);

      boolean resultSuccess;
      String stackTrace;
      if (result == 0) {
        RunResult runResult = streamConsumer.getRunResult();

        for (ResultPrinter printer : resultPrinterFactory.create()) {
          printer.print(runResult);
        }

        runResult.getApplicationCoverageReport().ifPresent(coverageLimitsChecker::setCoverageReport);
        saveRunDataToFile(runResult);

        resultSuccess = !runResult.hasFailed();
        stackTrace = runResult.getStackTrace();
      } else {
        throw new MojoExecutionException("Build Fail", new MojoExecutionException(errorStreamConsumer.getOutput()));
      }

      if (!resultSuccess) {
        throw exceptionFactory.buildException("Build Fail", new MojoExecutionException("MUnit Tests Failed\n" + stackTrace));
      }

      if (coverage != null && coverage.isRunCoverage() && !EE.value().equals(runtimeProduct)) {
        getLog().warn("Coverage is a EE only feature and you've selected to run over CE");
      }

      if (coverageLimitsChecker.failBuild()) {
        throw new MojoExecutionException("Build Fail", new MojoFailureException("Coverage limits were not reached"));
      }

    } catch (IOException | CommandLineException e) {
      e.printStackTrace();
    }
  }

  private void saveRunDataToFile(RunResult runResult) {
    saveCoverageConfigDataToFile(coverage == null ? new Coverage() : coverage);
    runResult.getApplicationCoverageReport().ifPresent(this::saveCoverageReportDataToFile);
  }

  private void saveCoverageReportDataToFile(ApplicationCoverageReport applicationCoverageReport) {
    saveAsJsonDataToFile(applicationCoverageReport, coverageReportDataFile);
  }

  private void saveCoverageConfigDataToFile(Coverage coverage) {
    saveAsJsonDataToFile(coverage, coverageConfigDataFile);
  }

  private File createAndGetMunitWorkingDirectory() {
    File workingDirectory =
        new File(project.getBuild().getDirectory(), MUNIT_WORKING_DIR.value() + "-" + Long.toString(System.nanoTime()));
    workingDirectory.mkdir();
    return workingDirectory;
  }

  protected void generateApplicationStructure() throws MojoExecutionException {
    File sourceFolder = Paths.get(project.getBuild().getDirectory()).toFile();
    try {
      getLog().debug("Attempting to create application structure source: " + sourceFolder.getAbsolutePath() + " - destination: "
          + munitWorkingDirectory.getAbsolutePath());
      new ApplicationStructureGenerator(sourceFolder, munitWorkingDirectory, project.getArtifactId()).generate();
    } catch (Exception e) {
      throw new MojoExecutionException("Fail to create application structure", e);
    }
  }

  protected JVMStarter createJVMStarter(RunConfiguration runConfiguration)
      throws MojoExecutionException, IOException {
    if (!munitTestsDirectory.exists()) {
      getLog().warn("The project has no " + munitTestsDirectory + " folder. Aborting MUnit test run.");
      return null;
    }
    getLog().debug("MUnit root folder found at: " + munitTestsDirectory.getAbsolutePath());

    String starterJarFileName = FileUtils.generateRandomFileName(STARTER_CLASS_FILE, ".jar");

    File jarFile = jarFileFactory.create(classpathManager.getEffectiveClasspath(), REMOTE_RUNNER_CLASS.getCanonicalName(),
                                         new File(project.getBuild().getDirectory()), starterJarFileName);

    JVMStarter jvmStarter = new JVMStarter(getLog())
        .withJVM(jvmLocator.locate())
        .withWorkingDirectory(getWorkingDirectory())
        .withJar(jarFile)
        .withArgLines(getEffectiveArgLines(starterJarFileName))
        .withSystemProperties(effectiveSystemProperties)
        .addEnvironmentVariables(environmentVariables);

    Map<String, File> fileArgLines = new HashMap<>();
    fileArgLines.put(RUN_CONFIGURATION_ARG, saveRunConfigurationToFile(runConfiguration));
    jvmStarter.withArgLines(fileArgLines);

    return jvmStarter;
  }

  protected File saveRunConfigurationToFile(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile =
        Paths.get(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), RUN_CONFIGURATION_JSON)
            .toFile();
    saveAsJsonDataToFile(runConfiguration, runConfigurationFile);
    return runConfigurationFile;
  }

  private void saveAsJsonDataToFile(Object data, File dataLocation) {
    try {
      if (dataLocation == null) {
        getLog().warn("Unable to save coverage data to a null location");
        return;
      }
      dataLocation.getParentFile().mkdirs();
      dataLocation.createNewFile();
      FileUtils.writeStringToFile(dataLocation, gson.toJson(data), Charset.defaultCharset());
      getLog().debug("Data File saved in " + dataLocation);
    } catch (IOException e) {
      getLog().warn("Unable to save data to file:" + e.getMessage());
      getLog().debug(e);
    }
  }

  protected List<String> getEffectiveArgLines(String starterJarFileName) {
    return new ArgLinesManager(argLines, starterJarFileName, getLog()).getEffectiveArgLines();
  }

  protected File getWorkingDirectory() {
    if (project != null) {
      return project.getBasedir();
    }
    return new File(".");
  }

  private Map<String, String> getEffectiveSystemProperties() {
    Map<String, String> effectiveSystemProperties = new HashMap<>();
    effectiveSystemProperties.putAll(getUserSystemProperties());
    // TODO MU-978
    effectiveSystemProperties.put("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER",
                                  "org.glassfish.grizzly.memory.HeapMemoryManager");
    return effectiveSystemProperties;
  }

  private Map<String, String> getUserSystemProperties() {
    return new UserPropertiesBuilder(project.getBuild().getDirectory(), getLog())
        .withSystemPropertyVariables(systemPropertyVariables)
        .withDynamicPorts(dynamicPorts)
        .withUserProperties(session != null ? session.getUserProperties() : null)
        .build();
  }

}
