/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.coverage.printer;

import java.util.Locale;

import org.apache.maven.plugin.logging.Log;

import org.mule.munit.remote.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.coverage.report.model.MuleFlow;
import org.mule.munit.remote.coverage.report.model.MuleResource;

/**
 * <p>
 * Prints Coverage Report through the Console.
 * </p>
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ConsolePrinter implements MavenCoverageReportPrinter {

  public static final String PRINTER_NAME = "Console";

  private Log log;

  public ConsolePrinter(Log log) {
    this.log = log;
  }

  public String getPrinterName() {
    return PRINTER_NAME;
  }


  public void printReport(ApplicationCoverageReport report) {
    log.info("===============================================================================");
    log.info("\t\t\t\tMUnit Coverage Report");
    log.info("===============================================================================");
    log.info("");
    for (MuleResource mr : report.getResources()) {
      log.info("-------------------------------------------------------------------------------");
      if (mr.getCoverage() != -1) {
        log.info("Resource File: " + mr.getName() + " - Flows: " + mr.getFlows().size() + " - Weight: "
            + formatDecimals(mr.getWeight()) + " -  Coverage: " + formatDecimals(mr.getCoverage()) + "%");
      } else {
        log.info("Resource File: " + mr.getName() + " - Flows: " + mr.getFlows().size() + " - Weight: "
            + formatDecimals(mr.getWeight()) + " -  Coverage: N/A");
      }
      log.info("-------------------------------------------------------------------------------");
      for (MuleFlow mf : mr.getFlows()) {
        log.info(
                 "  * Flow:" + mf.getName() + " - Covered (MP): " + mf.getCoveredLocations().size() + " - Total (MP): "
                     + mf.getLocations().size() + " - Coverage: " + formatDecimals(mf.getCoverage()) + "%");
      }
    }
    log.info("");
    log.info("===============================================================================");
    log.info("\t\t\t\t\tSummary");
    log.info("===============================================================================");
    log.info("  * CMPs: " + report.getApplicationCoveredMessageProcessorCount());
    log.info("  * MPs: " + report.getApplicationMessageProcessorCount());
    log.info("  * Flows: " + report.getApplicationFlowCount());
    log.info("  * Resources: " + report.getResources().size());
    log.info("===============================================================================");
    if (report.getApplicationFlowCount() > 0) {
      log.info("\t\t\t\t** Application Coverage: " + formatDecimals(report.getCoverage()) + "% **");
    } else {
      log.info("\t\t\t\t** Application Coverage: N/A **");
    }
    log.info("===============================================================================");

  }

  private String formatDecimals(Double number) {
    return String.format(Locale.ENGLISH, "%.2f", number);
  }

}
