/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mojo;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.maven.plugin.logging.Log;
import org.mule.munit.Coverage;
import org.mule.munit.coverage.CoverageAuthorizator;
import org.mule.munit.runner.printer.CoverageSummaryPrinter;
import org.mule.munit.runner.printer.MavenResultPrinter;
import org.mule.munit.runner.printer.ResultPrinter;
import org.mule.munit.runner.printer.TestOutputFilePrinter;
import org.mule.munit.runner.printer.TextResultFilePrinter;
import org.mule.munit.runner.printer.XmlResultPrinter;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * <p>
 * Returns the result printers based on the user configuration
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ResultPrinterFactory {

  private Log log;

  private File testOutputDirectory;
  private File surefireReportsFolder;
  private Map<String, String> systemProperties;
  private Coverage coverage;

  private boolean enableSurefireReports;
  private boolean redirectTestOutputToFile;
  private boolean enableCoverageReport;


  public ResultPrinterFactory(Log log) {
    this.log = log;
  }

  public ResultPrinterFactory withSurefireReports(Boolean enableSurefireReports, File surefireReportsFolder,
                                                  Map<String, String> systemProperties) {
    this.surefireReportsFolder = surefireReportsFolder;
    this.enableSurefireReports = enableSurefireReports;
    this.systemProperties = systemProperties;
    return this;
  }

  public ResultPrinterFactory withTestOutputReports(Boolean redirectTestOutputToFile, File testOutputDirectory) {
    checkNotNull(redirectTestOutputToFile, "Redirect Output to File property cannot be null");
    checkNotNull(testOutputDirectory, "Test Output Directory cannot be null");
    this.redirectTestOutputToFile = redirectTestOutputToFile;
    this.testOutputDirectory = testOutputDirectory;
    return this;
  }

  public ResultPrinterFactory withCoverageSummaryReport(Coverage coverage, String runtimeProduct) {
    this.coverage = coverage;
    this.enableCoverageReport = new CoverageAuthorizator(coverage, runtimeProduct).shouldRunCoverage();
    return this;
  }

  public List<ResultPrinter> create() {
    List<ResultPrinter> printers = new ArrayList<>();

    if (enableCoverageReport) {
      printers.add(new CoverageSummaryPrinter(log, coverage));
    }

    printers.add(new MavenResultPrinter(log));

    if (enableSurefireReports) {
      printers.addAll(getSurefireResultPrinters(redirectTestOutputToFile));
    }

    if (redirectTestOutputToFile) {
      printers.add(new TestOutputFilePrinter(testOutputDirectory, log));
    }


    return printers;
  }

  private List<ResultPrinter> getSurefireResultPrinters(Boolean redirectTestOutputToFile) {
    List<ResultPrinter> printers = new ArrayList<>();

    printers.add(new XmlResultPrinter(surefireReportsFolder, systemProperties, log));
    printers.add(new TextResultFilePrinter(surefireReportsFolder, log));
    if (redirectTestOutputToFile) {
      printers.add(new TestOutputFilePrinter(surefireReportsFolder, log));
    }

    return printers;
  }

}
