/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.consumer;

import static org.mule.munit.common.protocol.message.MessageField.FIELD_TOKEN;
import static org.mule.munit.common.protocol.message.MessageField.NEW_LINE_ESCAPE_TOKEN;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.shared.utils.cli.StreamConsumer;

import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.RunMessageParser;
import org.mule.munit.common.protocol.message.TestStatus;
import org.mule.munit.runner.handler.RunnerMessageHandler;
import org.mule.munit.runner.model.RunResult;

/**
 * <p>
 * A Stream Consumer that reads the Standard Output and identifies the Test Run protocol
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerStreamConsumer implements StreamConsumer {

  private RunResult runResult = new RunResult();
  private RunnerMessageHandler runnerMessageHandler;
  private RunMessageParser runnerParser;

  public RunnerStreamConsumer(RunnerMessageHandler runnerMessageHandler) {
    this.runnerMessageHandler = runnerMessageHandler;
    this.runnerParser = new RunMessageParser(new RunStreamConsumerEventListener());
  }

  public void consumeLine(String s) {
    if (s.startsWith(FIELD_TOKEN)) {
      handleProtocolLine(s.replace(FIELD_TOKEN, StringUtils.EMPTY).replaceAll(NEW_LINE_ESCAPE_TOKEN, System.lineSeparator()));
    } else {
      runnerMessageHandler.handleStandardOutputLine(s);
    }
  }

  protected void handleProtocolLine(String message) {
    runnerParser.parseAndNotify(message);
  }

  private class RunStreamConsumerEventListener implements RunEventListener {

    private String currentSuite;

    @Override
    public void notifyRunFinish() {
      runResult.setRunFinished(true);
    }

    @Override
    public void notifyCoverageReport(String coverageReportJson) {
      runResult.setCoverageReport(coverageReportJson);
    }

    @Override
    public void notifySuiteStart(String suite, int numberOfTests) {
      currentSuite = suite;
      runResult.testSuiteStarted(suite);
      runnerMessageHandler.handleSuiteStart(suite);
    }

    @Override
    public void notifyContainerFailure(String suite, String stackTrace) {
      runResult.testSuiteStartFailed(suite, stackTrace);
      runnerMessageHandler.handleSuiteStartFailure(runResult.getSuite(suite));
    }

    @Override
    public void notifyBeforeSuiteEnd(String name, String stackTrace, TestStatus status) {
      notifyBeforeOrAfterSuite(currentSuite, stackTrace, status);
    }

    @Override
    public void notifyTestStart(String name) {
      runResult.testStarted(currentSuite, name);
    }

    @Override
    public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
      switch (status) {
        case SUCCESS:
          runResult.testFinished(currentSuite, name, elapsedTime);
          break;
        case FAILURE:
          runResult.testFailure(currentSuite, name, stackTrace, elapsedTime);
          break;
        case IGNORED:
          runResult.testIgnored(currentSuite, name, elapsedTime);
          break;
        case ERROR:
          runResult.testError(currentSuite, name, stackTrace, elapsedTime);
          break;
      }
    }

    @Override
    public void notifyAfterSuiteEnd(String name, String stackTrace, TestStatus status) {
      notifyBeforeOrAfterSuite(currentSuite, stackTrace, status);
    }

    @Override
    public void notifySuiteEnd(String suite, long elapsedTime) {
      runResult.suiteFinished(suite, elapsedTime);
      runnerMessageHandler.handleSuiteFinished(runResult.getSuite(currentSuite));

    }

    @Override
    public void notifyUnexpectedError(String stackTrace) {
      runResult.unexpectedError(stackTrace);
      runnerMessageHandler.handleUnexpectedError(stackTrace);
    }

    @Override
    public void notifySuiteUnexpectedError(String stackTrace) {
      runResult.unexpectedError(stackTrace);
      runnerMessageHandler.handleUnexpectedError(stackTrace);
    }

  }

  public RunResult getRunResult() {
    runnerMessageHandler.handleRunResult(runResult);
    return runResult;
  }

  private void notifyBeforeOrAfterSuite(String currentSuite, String stackTrace, TestStatus status) {
    switch (status) {
      case ERROR:
        runResult.suiteError(currentSuite, stackTrace);
        runnerMessageHandler.handleSuiteFailure(runResult.getSuite(currentSuite));
        break;
      case FAILURE:
        runResult.suiteFailure(currentSuite, stackTrace);
        runnerMessageHandler.handleSuiteFailure(runResult.getSuite(currentSuite));
        break;
    }
  }

}
