/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.coverage.printer;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.List;

import org.mule.munit.Coverage;
import org.mule.munit.coverage.printer.model.CoverageReportModel;
import org.mule.munit.coverage.printer.model.MuleFlowModel;
import org.mule.munit.coverage.printer.model.MuleResourceModel;
import org.mule.munit.remote.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.coverage.report.model.MuleFlow;
import org.mule.munit.remote.coverage.report.model.MuleResource;

import com.google.gson.GsonBuilder;

/**
 * <p>
 * Generates JSON file Coverage Report.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class JsonPrinter implements MavenCoverageReportPrinter {

  public static final String PRINTER_NAME = "Json";
  private final File reportFile;
  private Coverage configuration;

  public JsonPrinter(Coverage configuration, File reportFile) {
    checkNotNull(reportFile, "The report path cannot be null");
    checkNotNull(configuration, "The coverage configuration cannot be null");
    this.configuration = configuration;
    this.reportFile = reportFile;
  }

  public String getPrinterName() {
    return PRINTER_NAME;
  }

  public void printReport(ApplicationCoverageReport report) {
    try {
      CoverageReportModel coverageReportModel = generateReportModel(report);
      reportFile.createNewFile();
      FileWriter fileWriter = new FileWriter(reportFile);
      new GsonBuilder().setPrettyPrinting().create().toJson(coverageReportModel, fileWriter);
      fileWriter.close();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  private CoverageReportModel generateReportModel(ApplicationCoverageReport report) {
    CoverageReportModel coverageReportModel = new CoverageReportModel(report, configuration);
    List<MuleResourceModel> files = new ArrayList<>();
    for (MuleResource resource : report.getResources()) {
      MuleResourceModel resourceModel = new MuleResourceModel(resource);
      List<MuleFlowModel> flows = new ArrayList<>();
      for (MuleFlow flow : resource.getFlows()) {
        flows.add(new MuleFlowModel(flow));
      }
      resourceModel.setFlows(flows);
      files.add(resourceModel);
    }
    coverageReportModel.setFiles(files);
    return coverageReportModel;
  }
}
