/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.reflection;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

/**
 * Builder to invoke methods by reflection.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MethodInvoker {

  private final Object instance;
  private final String methodName;

  private final List<Object> values = new ArrayList<>();
  private final List<Class<?>> types = new ArrayList<>();

  public MethodInvoker(Object instance, String methodName) {
    Preconditions.checkNotNull(instance, "The object instance must not be null");
    Preconditions.checkArgument(StringUtils.isNotBlank(methodName), "The method name must not be null nor empty");

    this.instance = instance;
    this.methodName = methodName;
  }

  public void clear() {
    this.types.clear();
    this.values.clear();
  }

  public MethodInvoker with(Class<?> type, Object value) {
    types.add(type);
    values.add(value);
    return this;
  }

  public Object invoke() throws ReflectionUtilsException {
    if (types.isEmpty() && values.isEmpty()) {
      return invokeInstanceNoParameters();
    } else {
      return invokeInstanceWithParameters();
    }
  }

  private Object invokeInstanceNoParameters() throws ReflectionUtilsException {
    Class<?> klass = instance.getClass();
    try {
      Method method = klass.getMethod(methodName);
      method.setAccessible(true);
      return method.invoke(instance);
    } catch (IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException
        | InvocationTargetException e) {
      throw new ReflectionUtilsException("Fail to invoke method " + klass.getCanonicalName() + "#" + methodName, e);
    }
  }

  private Object invokeInstanceWithParameters() throws ReflectionUtilsException {
    Class<?> klass = instance.getClass();
    try {
      Method method = klass.getMethod(methodName, getTypesArray(types));
      return method.invoke(instance, getParametersArray(values));
    } catch (IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException
        | InvocationTargetException e) {
      throw new ReflectionUtilsException("Fail to invoke method " + klass.getCanonicalName() + "#" + methodName, e);
    }
  }

  private Class<?>[] getTypesArray(List<? extends Class<?>> listOfTypes) {
    int i = 0;
    Class<?>[] types = new Class<?>[listOfTypes.size()];
    for (Class c : listOfTypes) {
      types[i] = c;
    }
    return types;
  }

  private Object[] getParametersArray(List<Object> listOfParameters) {
    int i = 0;
    Object[] types = new Object[listOfParameters.size()];
    for (Object c : listOfParameters) {
      types[i] = c;
    }
    return types;
  }
}
