/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;


import org.apache.commons.lang3.Validate;
import org.apache.maven.artifact.Artifact;

import java.util.StringTokenizer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Class in charge of creating Maven coordinates filters that can match attributes of a Maven Artifact
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class CoordinateFilter {

  public static final String MAVEN_COORDINATES_SEPARATOR = ":";
  public static final String WILD_CARD = "*";
  public static final int MAX_TOKENS = 5;

  private String groupId = WILD_CARD;
  private String artifactId = WILD_CARD;
  private String extension = WILD_CARD;
  private String classifier = WILD_CARD;
  private String version = WILD_CARD;

  private CoordinateFilter() {}

  /**
   * Receives a String that represents a coordinate to be matched and returns the proper filter for it, if the coordinate is
   * incomplete, it completes the remaining attributes with wild cards '*'
   *
   * @param coordinate Maven Coordinate string to be parsed Format: <groupId>:<artifactId>:<extension>:<classifier>:<version>
   * @return The filter that matches the received coordinate
   */
  public static CoordinateFilter of(String coordinate) {
    StringTokenizer stringTokenizer = new StringTokenizer(coordinate, MAVEN_COORDINATES_SEPARATOR);
    if (stringTokenizer.countTokens() > MAX_TOKENS) {
      throw new IllegalArgumentException(coordinate
          + " is invalid: Maven coordinates should be in the form of "
          + "<groupId>:<artifactId>:<extension>:<classifier>:<version>");
    }
    return new CoordinateFilterBuilder()
        .withGroupId(getNextToken(stringTokenizer))
        .withArtifactId(getNextToken(stringTokenizer))
        .withExtension(getNextToken(stringTokenizer))
        .withClassifier(getNextToken(stringTokenizer))
        .withVersion(getNextToken(stringTokenizer)).build();
  }

  /**
   * Returns true if the coordinate filter matches the Artifact attributes. If at least one of the attributes does not match, it
   * returns false
   *
   * @param artifact The artifact to be compared
   * @return If the artifact matches the filter
   */
  public boolean accepts(Artifact artifact) {
    String[] tokens =
        new String[] {artifact.getGroupId(), artifact.getArtifactId(), artifact.getType(), artifact.getClassifier(),
            artifact.getVersion()};

    String[] patternTokens = getPatternTokens();
    boolean matched = true;
    for (int i = 0; matched && i < tokens.length; i++) {
      matched = matches(tokens[i], patternTokens[i]);
    }
    return matched;
  }

  /**
   * Returns the coordinate format followed by Mule, which follows the following convention:
   * <groupId>:<artifactId>:<extension>:<classifier>:<version> Where the '*' symbol acts as a wild card
   *
   * @return The coordinate in the Mule format
   */
  public String getMuleCoordinatesFormat() {
    return Stream.of(getPatternTokens()).collect(Collectors.joining(MAVEN_COORDINATES_SEPARATOR));
  }

  private boolean matches(String token, String pattern) {
    if (WILD_CARD.equals(pattern)) {
      return true;
    }
    return pattern.equals(token);
  }

  private String[] getPatternTokens() {
    return new String[] {groupId, artifactId, extension, classifier, version};
  }

  private static String getNextToken(StringTokenizer stringTokenizer) {
    return stringTokenizer.hasMoreTokens() ? stringTokenizer.nextToken() : WILD_CARD;
  }

  /**
   * Coordinate Filter Builder
   *
   * @author Mulesoft Inc.
   * @since 2.0.0
   */
  public static class CoordinateFilterBuilder {

    private static final String GROUP_ID = "group id";
    private static final String ARTIFACT_ID = "artifact id";
    private static final String VERSION = "version";
    private static final String CLASSIFIER = "classifier";
    private static final String EXTENSION = "extension";

    private CoordinateFilter coordinateFilter = new CoordinateFilter();

    public CoordinateFilterBuilder withGroupId(String groupId) {
      validateIsNotNull(groupId, GROUP_ID);
      coordinateFilter.groupId = groupId;
      return this;
    }

    public CoordinateFilterBuilder withArtifactId(String artifactId) {
      validateIsNotNull(artifactId, ARTIFACT_ID);
      coordinateFilter.artifactId = artifactId;
      return this;
    }

    public CoordinateFilterBuilder withClassifier(String classifier) {
      validateIsNotNull(classifier, CLASSIFIER);
      coordinateFilter.classifier = classifier;
      return this;
    }

    public CoordinateFilterBuilder withVersion(String version) {
      validateIsNotNull(version, VERSION);
      coordinateFilter.version = version;
      return this;
    }

    public CoordinateFilterBuilder withExtension(String extension) {
      validateIsNotNull(extension, EXTENSION);
      coordinateFilter.extension = extension;
      return this;
    }

    public CoordinateFilter build() {
      return this.coordinateFilter;
    }

    private void validateIsNotNull(String value, String fieldId) {
      Validate.notNull(value, "Coordinate field '%s' cannot be null", fieldId);
    }

  }


}
