/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util.properties;

import static java.util.Arrays.asList;
import static org.mule.munit.remote.classloading.ClassLoaderUtils.STARTER_JAR_FILE_NAME;
import static org.mule.munit.util.ArgLinesManager.JAVA_SYSTEM_CLASS_LOADER;

import java.util.*;

import org.apache.maven.plugin.logging.Log;
import org.mule.munit.common.util.FreePortFinder;
import org.mule.munit.remote.RemoteRunner;

/**
 * <p>
 * Generates the final properties for the new JVM
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class UserPropertiesBuilder extends PropertiesFileBuilder {

  private static final int MIN_PORT_NUMBER = 40000;
  private static final int MAX_PORT_NUMBER = 50000;
  private static final Collection<String> INVALID_SYSTEM_PROPERTIES =
      asList("java.library.path", "file.encoding", "jdk.map.althashing.threshold", JAVA_SYSTEM_CLASS_LOADER,
             STARTER_JAR_FILE_NAME);


  public static final String MULE_TESTING_MODE = "mule.testingMode";
  public static final String MUNIT_RUNNER_SERVER_PORT = "munit.server.port";

  public static final String USER_SYSTEM_PROPERTIES_FILE = RemoteRunner.SYSTEM_PROPERTIES_FILE;
  public static final String USER_SYSTEM_PROPERTIES_FILE_NAME = "user.system.properties";

  private Map<String, String> systemPropertyVariables = Collections.emptyMap();
  private Properties userProperties;
  private List<String> dynamicPorts = Collections.emptyList();

  public UserPropertiesBuilder(String propertiesFileFolderPath, Log log) {
    super(propertiesFileFolderPath, log);
  }

  public UserPropertiesBuilder withSystemPropertyVariables(Map<String, String> systemPropertyVariables) {
    this.systemPropertyVariables = systemPropertyVariables;
    return this;
  }

  public UserPropertiesBuilder withDynamicPorts(List<String> dynamicPorts) {
    this.dynamicPorts = dynamicPorts;
    return this;
  }

  public UserPropertiesBuilder withUserProperties(Properties userProperties) {
    this.userProperties = userProperties;
    return this;
  }

  public Map<String, String> build() {
    Map<String, String> effectiveSystemProperties = new HashMap<>();

    setSystemPropertyVariables(effectiveSystemProperties);
    setDynamicPorts(effectiveSystemProperties);
    setMunitRunnerServer(effectiveSystemProperties);
    setStopLicenseCheck(effectiveSystemProperties);
    setUserSystemProperties(effectiveSystemProperties);
    removeInvalidSystemProperties(effectiveSystemProperties);

    effectiveSystemProperties = savePropertiesFileIfRequired(effectiveSystemProperties);

    return effectiveSystemProperties;
  }

  @Override
  protected String getFileProperty() {
    return USER_SYSTEM_PROPERTIES_FILE;
  }

  @Override
  protected String getFileName() {
    return USER_SYSTEM_PROPERTIES_FILE_NAME;
  }


  private void setStopLicenseCheck(Map<String, String> props) {
    log.debug("Avoid license check for Mule EE components...");
    props.put(MULE_TESTING_MODE, "true");
  }

  private void setMunitRunnerServer(Map<String, String> props) {
    FreePortFinder portFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
    Integer port = portFinder.find();
    log.debug("Setting port for the MUnit runner server : " + port);
    props.put(MUNIT_RUNNER_SERVER_PORT, String.valueOf(port));
  }

  private void setSystemPropertyVariables(Map<String, String> systemProperties) {
    if (systemPropertyVariables != null) {
      systemProperties.putAll(systemPropertyVariables);
      log.debug("Adding System Property Variables : " + systemPropertyVariables);
    }
  }

  private void setUserSystemProperties(Map<String, String> systemProperties) {
    if (userProperties != null) {
      for (Map.Entry<Object, Object> prop : userProperties.entrySet()) {
        Object key = prop.getKey();
        Object value = prop.getValue();
        if (key instanceof String && value instanceof String) {
          systemProperties.put(prop.getKey().toString(), prop.getValue().toString());
          log.debug(String.format("Setting System Property [%s] to %s", key, value));
        }
      }
    }
  }

  private void setDynamicPorts(Map<String, String> systemProperties) {
    if (dynamicPorts != null) {
      log.info("Acquiring dynamic ports...");
      FreePortFinder portFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
      for (String portPlaceHolder : dynamicPorts) {
        Integer dynamicPort = portFinder.find();
        systemProperties.put(portPlaceHolder, dynamicPort.toString());
        log.debug(String.format("Dynamic port [%s] set to: [%s]", portPlaceHolder, dynamicPort));
      }
      log.info("Dynamic port definition [DONE]");
    }
  }

  private void removeInvalidSystemProperties(Map<String, String> effectiveSystemProperties) {
    INVALID_SYSTEM_PROPERTIES.stream().filter(effectiveSystemProperties::containsKey).forEach(invalidProp -> {
      effectiveSystemProperties.remove(invalidProp);
      log.warn(invalidProp + " cannot be set as system property, use <argLine>-D"
          + invalidProp + "=...</argLine> instead");
    });
  }

}
