/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mojo;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * The class handles the filtering logic for the MUnit test suite files
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class TestSuiteFileFilter {

  private static final String SINGLE_TEST_NAME_TOKEN = "#";

  private final Log log;

  private String testNameRegEx;
  private String testSuiteRegEx;

  public TestSuiteFileFilter(Log log, String filter) {
    checkNotNull(log, "The log must not be null");

    this.log = log;
    if (StringUtils.isNotBlank(filter) && filter.contains(SINGLE_TEST_NAME_TOKEN)) {
      testSuiteRegEx = filter.substring(0, filter.indexOf(SINGLE_TEST_NAME_TOKEN));
      testNameRegEx = filter.substring(filter.indexOf(SINGLE_TEST_NAME_TOKEN) + 1);
    } else {
      testSuiteRegEx = filter;
      testNameRegEx = "";
    }
  }

  public String getTestNameRegEx() {
    return testNameRegEx;
  }

  /**
   * Evaluates if the provided file name matches the pattern.
   *
   * @param fileName
   * @return true the fileName matches the filter regular expression or if there is no regular expression filter
   */
  public boolean shouldFilter(String fileName) {
    checkArgument(StringUtils.isNotBlank(fileName), "The fileName should not be null nor empty");

    if (StringUtils.isBlank(testSuiteRegEx)) {
      return true;
    }

    boolean result = fileName.matches(testSuiteRegEx);
    if (!result) {
      log.debug("MUnit Test Suite file " + fileName + " skipped. It doesn't match filter criteria: [" + testSuiteRegEx + "]");
      return false;
    }
    return true;
  }



}
