/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import static java.util.Arrays.asList;
import static org.mule.munit.remote.classloading.ClassLoaderUtils.STARTER_JAR_FILE_NAME;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;

/**
 * <p>
 * Generates the final argument lines for the new JVM
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ArgLinesManager {

  public static final String JAVA_SYSTEM_CLASS_LOADER = "java.system.class.loader";
  public static final String MUNIT_DEBUG_DEFAULT_SYSTEM_PROPERTY = "munit.debug.default";

  private static final List<String> INVALID_ARG_LINES =
      asList("-jar", "-cp", "-classpath", "-D" + STARTER_JAR_FILE_NAME, "-D" + JAVA_SYSTEM_CLASS_LOADER);

  private static final String DEFAULT_DEBUG_ARG_LINE = "-agentlib:jdwp=transport=dt_socket,server=y,address=8002,suspend=y";

  private Log log;
  private String starterJarFileName;
  private List<String> originalArgLines;

  public ArgLinesManager(List<String> originalArgLines, String starterJarFileName, Log log) {
    this.originalArgLines = originalArgLines;
    this.starterJarFileName = starterJarFileName;
    this.log = log;
  }

  public List<String> getEffectiveArgLines() {
    List<String> effectiveArgLines = new ArrayList<>();
    for (String argLine : originalArgLines) {
      if (filterArgLine(argLine)) {
        log.warn(argLine + " can not be set. We will ignore this argLine");
      } else {
        effectiveArgLines.add(argLine);
      }
    }
    addDefaultArgLines(effectiveArgLines);

    return effectiveArgLines;
  }

  private boolean filterArgLine(String argLine) {
    if (StringUtils.isBlank(argLine)) {
      return true;
    }

    for (String invalidArgLine : INVALID_ARG_LINES) {
      if (argLine.startsWith(invalidArgLine)) {
        return true;
      }
    }
    return false;
  }

  private void addDefaultArgLines(List<String> effectiveArgLines) {
    effectiveArgLines.add("-D" + STARTER_JAR_FILE_NAME + "=" + starterJarFileName);

    if (Boolean.valueOf(System.getProperty(MUNIT_DEBUG_DEFAULT_SYSTEM_PROPERTY, "false"))) {
      effectiveArgLines.add(DEFAULT_DEBUG_ARG_LINE);
    }
  }

}

