/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util.maven;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.tools.api.packager.packaging.Classifier;

import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.project.MavenProject;

/**
 * It knows how to search the Maven API to find specific dependencies that have a meaning for a Mule Application
 * 
 * @author Mulesoft Inc.
 * @since 2.1.0
 */
public class MuleApplicationDependencyFinder {

  private MavenProject project;

  public MuleApplicationDependencyFinder(MavenProject project) {
    checkArgument(project != null, "The project must not be null");
    this.project = project;
  }

  /**
   * Looks for a mule domain artifact amongst the project's artifacts, provided a mule domain is declared as dependency in the
   * project.
   *
   * @return an optional with the dependency, empty if none find
   */
  public Optional<Artifact> findMuleDomainArtifact() {
    Optional<Dependency> domainDependency = findMuleDomainDependency();
    if (domainDependency.isPresent()) {
      return findMuleDomainArtifact(buildDependencyId(domainDependency.get()));
    }
    return Optional.empty();
  }

  /**
   * Looks for a mule domain artifact amongst the project's artifacts that matches the artifact id provided project.
   *
   * @return an optional with the dependency, empty if none find
   */
  public Optional<Artifact> findMuleDomainArtifact(String artifactId) {
    checkArgument(isNotBlank(artifactId), "The artifact id must not be null nor empty");

    Optional<Artifact> artifact = project.getArtifacts().stream()
        .filter(a -> a.getId().equals(artifactId))
        .findFirst();
    return artifact;
  }

  /**
   * Looks for a mule domain dependency amongst the declared domains of the project.
   * 
   * @return an optional with the dependency, empty if none find
   */
  public Optional<Dependency> findMuleDomainDependency() {
    Optional<Dependency> dependency = project.getDependencies().stream()
        .filter(d -> StringUtils.isBlank(d.getClassifier()) ? false : d.getClassifier().equals(Classifier.MULE_DOMAIN.toString()))
        .findFirst();
    return dependency;
  }

  /**
   * It builds a generic ID based on a dependency
   * 
   * @param dependency the dependency to build the ID for
   * @return a string representing the coordinates of the ID
   */
  public static String buildDependencyId(Dependency dependency) {
    checkArgument(dependency != null, "The dependency must not be null");

    StringBuilder dependencyId = new StringBuilder();
    dependencyId
        .append(dependency.getGroupId()).append(":")
        .append(dependency.getArtifactId()).append(":")
        .append(dependency.getType()).append(":");

    if (isNotBlank(dependency.getClassifier())) {
      dependencyId.append(dependency.getClassifier());
      dependencyId.append(":");
    }
    dependencyId.append(dependency.getVersion());
    return dependencyId.toString();
  }
}
