/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.munit.MUnitMojo.MULE_ARTIFACT_JSON_FILE_NAME;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.mule.munit.common.util.FileUtils;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;

/**
 * Reads the mule application model and gives access to its properties based on the user settings
 * 
 * @since 2.0.0
 * @author Mulesoft Inc.
 */
public class MuleApplicationModelLoader {

  private MuleApplicationModel muleApplicationModel;
  private String runtimeVersion;
  private String runtimeProduct;
  private Log log;

  public MuleApplicationModelLoader(File muleApplicationJsonFile, String runtimeVersion, String runtimeProduct, Log log)
      throws MojoExecutionException {
    try {
      this.muleApplicationModel = getMuleApplicationModel(muleApplicationJsonFile);
    } catch (IOException e) {
      String message = "Fail to read mule application file from " + muleApplicationJsonFile;
      log.error(message, e);
      throw new MojoExecutionException(message, e);
    }
    this.runtimeVersion = runtimeVersion;
    this.runtimeProduct = runtimeProduct;
    this.log = log;
  }

  public String getRuntimeVersion() {
    if (isNotBlank(runtimeVersion)) {
      return runtimeVersion;
    }
    String runtimeVersion = muleApplicationModel.getMinMuleVersion();
    log.debug("Runtime version set to " + runtimeVersion + " obtained from the " + MULE_ARTIFACT_JSON_FILE_NAME + " file");
    return runtimeVersion;
  }

  public String getRuntimeProduct() {
    if (isNotBlank(runtimeProduct)) {
      return runtimeProduct;
    }
    String runtimeProduct = muleApplicationModel.getRequiredProduct().name();
    log.debug("Runtime product set to " + runtimeProduct + " obtained from the " + MULE_ARTIFACT_JSON_FILE_NAME + " file");
    return runtimeProduct;
  }

  private MuleApplicationModel getMuleApplicationModel(File muleArtifactJsonFile) throws IOException {
    return new MuleApplicationModelJsonSerializer()
        .deserialize(FileUtils.readFileToString(muleArtifactJsonFile, Charset.defaultCharset()));
  }
}
