/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

/**
 * This class knows how to access several properties file that belongs (by default) to a Mule application.
 * It also provide specific methods to retrieve particular propreties.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MulePropertiesLoader {

  private static final String MULE_DEPLOY_PROPERTIES = "mule-deploy.properties";

  private transient Log log;
  private String classesPath;
  private String testClassesPath;

  public MulePropertiesLoader(String classesPath, String testClassesPath, Log log) {
    this.log = log;
    this.classesPath = classesPath;
    this.testClassesPath = testClassesPath;
  }

  /**
   * It'll return the value of the property config.resources in the file mule-deploy.properties
   *
   * @return the config resources or an empty string.
   */
  public String getConfigResources() {
    Properties props = this.loadProperties();

    if (props != null && props.getProperty("config.resources") != null) {
      return props.getProperty("config.resources");
    } else {
      return "";
    }
  }

  private Properties loadProperties() {
    try {
      Properties prop = new Properties();

      URL muleDeployPropertiesUrl = findMuleDeployProperties();
      if (null == muleDeployPropertiesUrl) {
        return null;
      }

      InputStream in = muleDeployPropertiesUrl.openStream();
      prop.load(in);
      in.close();
      return prop;
    } catch (Throwable t) {
      return null;
    }
  }

  private URL findMuleDeployProperties() throws IOException {
    URL muleDeployPropertiesUrl = null;
    File classesFile = new File(classesPath, MULE_DEPLOY_PROPERTIES);
    File testClassesFile = new File(testClassesPath, MULE_DEPLOY_PROPERTIES);

    if (classesFile.exists()) {
      muleDeployPropertiesUrl = classesFile.toURI().toURL();
    } else if (testClassesFile.exists()) {
      muleDeployPropertiesUrl = testClassesFile.toURI().toURL();
    }

    if (null == muleDeployPropertiesUrl) {
      log.debug("Failed to find " + MULE_DEPLOY_PROPERTIES + " file");
    } else {
      log.debug("Found " + MULE_DEPLOY_PROPERTIES + " file at: " + muleDeployPropertiesUrl.getPath());
    }

    return muleDeployPropertiesUrl;
  }


}
