/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Sets.newHashSet;
import static java.util.Collections.emptySet;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

import org.mule.munit.Coverage;
import org.mule.munit.coverage.CoverageAuthorizator;
import org.mule.munit.mojo.TestSuiteFileFilter;
import org.mule.munit.mojo.locators.Log4J2ConfigurationLocator;
import org.mule.munit.mojo.locators.TestSuiteFilesLocator;
import org.mule.munit.remote.api.configuration.ContainerConfiguration;
import org.mule.munit.remote.api.configuration.ContainerConfiguration.ContainerConfigurationBuilder;
import org.mule.munit.remote.api.configuration.CoverageConfiguration;
import org.mule.munit.remote.api.configuration.MavenConfiguration;
import org.mule.munit.remote.api.configuration.NotifierConfiguration;
import org.mule.munit.remote.api.configuration.NotifierParameter;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.ServerPluginConfiguration;
import org.mule.munit.remote.api.configuration.ServerPluginConfiguration.ServerPluginConfigurationBuilder;
import org.mule.munit.remote.notifiers.ConsoleNotifier;
import org.mule.munit.runner.structure.WorkingDirectoryGenerator;

/**
 * Creates {@link RunConfiguration}
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunConfigurationFactory {

  public static final String RUN_TOKEN_CONSOLE_PARAMETER = "runToken";

  private Log log;
  private String munitTags;
  private Boolean skipAfterFailure;
  private TestSuiteFileFilter testSuiteFileFilter;
  private WorkingDirectoryGenerator workingDirectoryGenerator;
  private MuleApplicationModelLoader muleApplicationModelLoader;


  private File munitSrcFolder;

  private Coverage coverage;
  private String pluginVersion;

  protected MavenProject project;
  protected MavenSession session;


  public RunConfigurationFactory(Log log, String munitTest, String munitTags, Boolean skipAfterFailure,
                                 MuleApplicationModelLoader muleApplicationModelLoader,
                                 WorkingDirectoryGenerator workingDirectoryGenerator,
                                 File munitSrcFolder, Coverage coverage, String pluginVersion,
                                 MavenProject project,
                                 MavenSession session) {

    checkNotNull(log, "The log must not be null");
    checkNotNull(skipAfterFailure, "The skipAfterFailure must not be null");
    checkNotNull(muleApplicationModelLoader, "The muleApplicationModelLoader must not be null nor empty");
    checkNotNull(workingDirectoryGenerator, "The WorkingDirectoryGenerator must not be null");
    checkNotNull(munitSrcFolder, "The munitSrcFolder must not be null");
    checkArgument(isNotBlank(pluginVersion), "The pluginVersion must not be null nor empty");
    checkNotNull(project, "The project must not be null");
    checkNotNull(session, "The session must not be null");

    this.log = log;
    this.munitTags = munitTags;
    this.skipAfterFailure = skipAfterFailure;
    this.muleApplicationModelLoader = muleApplicationModelLoader;
    this.testSuiteFileFilter = new TestSuiteFileFilter(log, munitTest);

    this.munitSrcFolder = munitSrcFolder;
    this.workingDirectoryGenerator = workingDirectoryGenerator;

    this.coverage = coverage;
    this.pluginVersion = pluginVersion;

    this.project = project;
    this.session = session;
  }

  public RunConfiguration createRunConfiguration() throws MojoExecutionException {
    String runToken = UUID.randomUUID().toString();

    List<NotifierParameter> consoleParameters = new ArrayList<>();
    consoleParameters.add(new NotifierParameter(RUN_TOKEN_CONSOLE_PARAMETER, String.class.getCanonicalName(), runToken));
    NotifierConfiguration consoleNotifierConfiguration = new NotifierConfiguration.NotifierConfigurationBuilder()
        .withClazz(ConsoleNotifier.class.getCanonicalName())
        .withParameters(consoleParameters)
        .build();

    List<NotifierConfiguration> notifierConfigurations = new ArrayList<>();
    notifierConfigurations.add(consoleNotifierConfiguration);

    MavenConfiguration mavenConfiguration = new MavenConfiguration.MavenConfigurationBuilder()
        .withMavenRepositoryDirectoryPath(session.getLocalRepository().getBasedir())
        .withSettingsXmlFilePath(session.getRequest().getUserSettingsFile().getAbsolutePath())
        .withGlobalSettingsXmlFilePath(session.getRequest().getGlobalSettingsFile().getAbsolutePath())
        .withForcePolicyUpdate(true)
        .withOfflineMode(session.isOffline())
        .withIgnoreArtifactDescriptorRepositories(false)
        .build();

    ContainerConfigurationBuilder containerConfigurationBuilder = new ContainerConfigurationBuilder()
        .withRuntimeId(muleApplicationModelLoader.getRuntimeVersion())
        .withProduct(muleApplicationModelLoader.getRuntimeProduct())
        .withMunitWorkingDirectoryPath(workingDirectoryGenerator.generateWorkingDirectory().toFile().getAbsolutePath())
        .withLog4JConfigurationFilePath(getLog4JConfigurationFilePath())
        .withMavenConfiguration(mavenConfiguration);

    Boolean shouldRunCoverage =
        new CoverageAuthorizator(coverage, muleApplicationModelLoader.getRuntimeProduct()).shouldRunCoverage();
    CoverageConfiguration coverageConfiguration = buildCoverageConfiguration(shouldRunCoverage);
    if (shouldRunCoverage) {
      String coveragePluginVersion = pluginVersion;
      ServerPluginConfiguration coveragePluginConfiguration = getCoverageServerPluginConfiguration(coveragePluginVersion);
      containerConfigurationBuilder.withServerPluginConfiguration(coveragePluginConfiguration);
    }

    ContainerConfiguration containerConfiguration = containerConfigurationBuilder.build();

    RunConfiguration.RunConfigurationBuilder builder = new RunConfiguration.RunConfigurationBuilder();

    String testToRunName = testSuiteFileFilter.getTestNameRegEx();

    builder.withRunToken(runToken)
        .withProjectName(project.getArtifactId())
        .withTags(munitTags == null ? Collections.emptySet() : new HashSet<>(Arrays.asList(munitTags.split(","))))
        .withSkipAfterFailure(skipAfterFailure)
        .withTestNames(isBlank(testToRunName) ? Collections.emptySet() : newHashSet(testToRunName.split(",")))
        .withSuitePaths(locateMunitTestSuites())
        .withNotifierConfigurations(notifierConfigurations)
        .withCoverageConfiguration(coverageConfiguration)
        .withContainerConfiguration(containerConfiguration);

    builder.withDomainLocation(workingDirectoryGenerator.generateDomainStructure()
        .map(domainPath -> domainPath.toFile().getAbsolutePath())
        .orElse(StringUtils.EMPTY));

    return builder.build();
  }

  private CoverageConfiguration buildCoverageConfiguration(Boolean shouldRunCoverage) {
    Set<String> coverageIgnoreFlows = coverage != null ? coverage.getIgnoreFlows() : emptySet();
    Set<String> coverageIgnoreFiles = coverage != null ? coverage.getIgnoreFiles() : emptySet();

    CoverageConfiguration coverageConfiguration = new CoverageConfiguration.CoverageConfigurationBuilder()
        .withShouldRunCoverage(shouldRunCoverage)
        .withSuitePaths(locateMunitTestSuites())
        .withIgnoredFiles(coverageIgnoreFiles)
        .withIgnoredFlowNames(coverageIgnoreFlows)
        .build();

    return coverageConfiguration;
  }

  private ServerPluginConfiguration getCoverageServerPluginConfiguration(String coveragePluginVersion) {
    return new ServerPluginConfigurationBuilder()
        .withGroupId("com.mulesoft.munit.plugins")
        .withArtifactId("munit-coverage-plugin")
        .withVersion(coveragePluginVersion)
        .withClassifier("mule-server-plugin")
        .withType("jar")
        .build();
  }

  private String getLog4JConfigurationFilePath() {
    return getLog4JConfigurationFile().isPresent() ? getLog4JConfigurationFile().get().getAbsolutePath() : EMPTY;
  }

  private Optional<File> getLog4JConfigurationFile() {
    List<File> files = new Log4J2ConfigurationLocator(log).locateFiles(new File(project.getBuild().getDirectory()));
    return files.isEmpty() ? Optional.empty() : Optional.of(files.get(0));
  }

  private Set<String> locateMunitTestSuites() {
    return new TestSuiteFilesLocator().locateFiles(munitSrcFolder).stream()
        .map(suiteFile -> munitSrcFolder.toURI().relativize(suiteFile.toURI()).toASCIIString())
        .filter(suitePath -> testSuiteFileFilter.shouldFilter(suitePath))
        .collect(Collectors.toSet());
  }

}
