/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static jersey.repackaged.com.google.common.collect.ImmutableMap.of;
import static org.mule.munit.plugin.maven.RuntimeProducts.EE;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.mule.munit.coverage.CoverageLimitsChecker;
import org.mule.munit.mojo.ApplicationResultPrinterFactory;
import org.mule.munit.mojo.exceptions.MojoExecutionExceptionFactory;
import org.mule.munit.plugin.maven.AbstractMunitMojo;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.util.MuleApplicationModelLoader;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.munit.remote.api.project.MuleApplicationStructureGenerator;
import org.mule.munit.util.ApplicationRunConfigurationFactory;


/**
 * <p>
 * MUnit Mojo to run tests
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
@Mojo(name = "test", defaultPhase = LifecyclePhase.TEST, requiresDependencyResolution = ResolutionScope.COMPILE)
public class MUnitMojo extends AbstractMunitMojo {

  @Parameter(property = "munit.coverage")
  protected Coverage coverage;

  @Parameter(defaultValue = "${munit.randomFailMessages}")
  private boolean randomFailMessages = false;

  @Parameter(defaultValue = "${plugin.version}")
  protected String pluginVersion;

  @Parameter(property = "munit.coverageReportData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-report.data")
  protected File coverageReportDataFile;

  @Parameter(property = "munit.coverageConfigData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-config.data")
  protected File coverageConfigDataFile;

  @Parameter(property = "runtimeVersion")
  public String runtimeVersion;

  @Parameter(property = "runtimeProduct")
  public String runtimeProduct;

  protected MojoExecutionExceptionFactory exceptionFactory;
  protected CoverageLimitsChecker coverageLimitsChecker;

  @Override
  protected void init() throws MojoExecutionException {
    super.init();
    this.exceptionFactory = new MojoExecutionExceptionFactory(randomFailMessages);
    this.coverageLimitsChecker = new CoverageLimitsChecker(coverage, muleApplicationModelLoader.getRuntimeProduct(), getLog());
  }

  @Override
  protected void handleRunResult(TargetRuntime targetRuntime, RunResult runResult) {
    super.handleRunResult(targetRuntime, runResult);
    runResult.getApplicationCoverageReport().ifPresent(coverageLimitsChecker::setCoverageReport);
    saveRunDataToFile(runResult);
  }

  @Override
  protected void failBuildIfNecessary(Map<TargetRuntime, RunResult> runResults) throws MojoExecutionException {
    try {
      super.failBuildIfNecessary(runResults);
    } catch (MojoExecutionException e) {
      throw exceptionFactory.buildException("Build Failed", e);
    }
    if (coverage != null && coverage.isRunCoverage() && !EE.value().equals(muleApplicationModelLoader.getRuntimeProduct())) {
      getLog().warn("Coverage is a EE only feature and you've selected to run over CE");
    }
    if (coverageLimitsChecker.failBuild()) {
      throw new MojoExecutionException("Build Failed", new MojoFailureException("Coverage limits were not reached"));
    }
  }

  @Override
  protected Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    TargetRuntime targetRuntime =
        new TargetRuntime(muleApplicationModelLoader.getRuntimeVersion(), muleApplicationModelLoader.getRuntimeProduct());
    return of(targetRuntime,
              new ApplicationRunConfigurationFactory(getLog(), munitTest, munitTags, skipAfterFailure, targetRuntime,
                                                     workingDirectoryGenerator, munitTestsDirectory, coverage, pluginVersion,
                                                     project, session).create());
  }

  @Override
  protected ResultPrinterFactory getResultPrinterFactory() {
    return new ApplicationResultPrinterFactory(getLog())
        .withCoverageSummaryReport(coverage, muleApplicationModelLoader.getRuntimeProduct())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory);
  }

  @Override
  protected ApplicationStructureGenerator getApplicationStructureGenerator() {
    return new MuleApplicationStructureGenerator(project.getBasedir().toPath(), Paths.get(project.getBuild().getDirectory()));
  }

  @Override
  protected File getMuleApplicationJsonPath() {
    Path projectBuildDirectoryPath = Paths.get(project.getBuild().getDirectory());
    Path muleApplicationJsonPath = projectBuildDirectoryPath
        .resolve(META_INF.value()).resolve(MULE_ARTIFACT.value()).resolve(MULE_ARTIFACT_JSON_FILE_NAME);
    return muleApplicationJsonPath.toFile();
  }

  @Override
  public MuleApplicationModelLoader getMuleApplicationModelLoader() throws MojoExecutionException {
    return super.getMuleApplicationModelLoader().withRuntimeVersion(runtimeVersion).withRuntimeProduct(runtimeProduct);
  }

  private void saveRunDataToFile(RunResult runResult) {
    saveAsJsonDataToFile(coverage == null ? new Coverage() : coverage, coverageConfigDataFile);
    runResult.getApplicationCoverageReport().ifPresent(this::saveCoverageReportDataToFile);
  }

  private void saveCoverageReportDataToFile(ApplicationCoverageReport applicationCoverageReport) {
    saveAsJsonDataToFile(applicationCoverageReport, coverageReportDataFile);
  }

}
