/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.coverage;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.apache.maven.plugin.logging.Log;

import org.mule.munit.Coverage;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;

/**
 * <p>
 * Defines if a build should fail based on the coverage configuration.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageLimitsChecker {

  private Log log;
  private Coverage coverageConfiguration;
  private CoverageAuthorizator coverageAuthorizator;

  private ApplicationCoverageReport coverageReport = null;

  public CoverageLimitsChecker(Coverage coverageConfiguration, String runtimeProduct, Log log) {
    checkNotNull(coverageConfiguration, "The coverage configuration must not be null");
    checkArgument(isNotBlank(runtimeProduct), "The runtime product must not be null nor empty");
    checkNotNull(log, "The log configuration must not be null");

    this.log = log;
    this.coverageConfiguration = coverageConfiguration;
    this.coverageAuthorizator = new CoverageAuthorizator(coverageConfiguration, runtimeProduct);
  }

  public void setCoverageReport(ApplicationCoverageReport coverageReport) {
    this.coverageReport = coverageReport;
  }

  public Boolean failBuild() {
    if (isCoverageActivated() && coverageConfiguration.getFailBuild()) {
      checkState(coverageReport != null, "No coverage report has been defined");
      return failsRequiredApplicationCoverage() || failsRequiredResourceCoverage() || failsRequiredFlowCoverage();
    }

    log.debug("Coverage is either not activated or not flagged to fail.");
    return false;
  }

  private boolean failsRequiredApplicationCoverage() {
    if (coverageConfiguration.getRequiredApplicationCoverage() == -1) {
      return false;
    }
    return coverageReport.getCoverage() != -1
        && coverageReport.getCoverage() < coverageConfiguration.getRequiredApplicationCoverage();
  }

  private boolean failsRequiredResourceCoverage() {
    if (coverageConfiguration.getRequiredResourceCoverage() == -1) {
      return false;
    }
    return coverageReport.getResources().stream()
        .anyMatch(mr -> mr.getCoverage() != -1 && mr.getCoverage() < coverageConfiguration.getRequiredResourceCoverage());
  }

  private boolean failsRequiredFlowCoverage() {
    if (coverageConfiguration.getRequiredFlowCoverage() == -1) {
      return false;
    }
    return coverageReport.getResources().stream().flatMap(mr -> mr.getFlows().stream())
        .anyMatch(mf -> mf.getCoverage() != -1 && mf.getCoverage() < coverageConfiguration.getRequiredFlowCoverage());
  }

  private Boolean isCoverageActivated() {
    return coverageAuthorizator.shouldRunCoverage();
  }

}
