/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static java.util.function.Function.identity;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

import static jersey.repackaged.com.google.common.collect.ImmutableMap.of;

import static org.mule.munit.plugin.maven.RuntimeProducts.EE;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;

import java.io.File;
import java.net.URI;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;

import org.mule.munit.coverage.CoverageLimitsChecker;
import org.mule.munit.mojo.ApplicationResultPrinterFactory;
import org.mule.munit.mojo.exceptions.MojoExecutionExceptionFactory;
import org.mule.munit.plugin.maven.AbstractMunitMojo;
import org.mule.munit.plugin.maven.TargetRuntime;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.munit.remote.api.project.MuleApplicationStructureGenerator;
import org.mule.munit.util.ApplicationRunConfigurationFactory;


/**
 * <p>
 * MUnit Mojo to run tests
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
@Mojo(name = "test", defaultPhase = LifecyclePhase.TEST, requiresDependencyResolution = ResolutionScope.COMPILE)
public class MUnitMojo extends AbstractMunitMojo {

  @Parameter(property = "munit.coverage")
  protected Coverage coverage;

  @Parameter(defaultValue = "${munit.randomFailMessages}")
  private boolean randomFailMessages = false;

  @Parameter(defaultValue = "${plugin.version}")
  protected String pluginVersion;

  @Parameter(property = "munit.coverageReportData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-report.data")
  protected File coverageReportDataFile;

  @Parameter(property = "munit.coverageConfigData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-config.data")
  protected File coverageConfigDataFile;

  protected MojoExecutionExceptionFactory exceptionFactory;
  protected CoverageLimitsChecker coverageLimitsChecker;

  @Override
  protected void init() throws MojoExecutionException {
    super.init();
    this.exceptionFactory = new MojoExecutionExceptionFactory(randomFailMessages);
    this.coverageLimitsChecker = new CoverageLimitsChecker(coverage, muleApplicationModelLoader.getRuntimeProduct(), getLog());
  }

  @Override
  protected void handleRunResult(TargetRuntime targetRuntime, RunResult runResult) {
    super.handleRunResult(targetRuntime, runResult);
    runResult.getApplicationCoverageReport().ifPresent(coverageLimitsChecker::setCoverageReport);
    saveRunDataToFile(runResult);
  }

  @Override
  protected void failBuildIfNecessary(Map<TargetRuntime, RunResult> runResults) throws MojoExecutionException {
    try {
      super.failBuildIfNecessary(runResults);

    } catch (MojoExecutionException cause) {
      throw exceptionFactory.buildException("Build Failed", cause);
    }

    if (coverage != null && coverage.isRunCoverage() && !EE.value().equals(muleApplicationModelLoader.getRuntimeProduct())) {
      getLog().warn("Coverage is a EE only feature and you've selected to run over CE");
    }

    if (coverageLimitsChecker.failBuild()) {
      throw new MojoExecutionException("Build Failed", new MojoFailureException("Coverage limits were not reached"));
    }
  }

  @Override
  protected Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    TargetRuntime targetRuntime = new TargetRuntime(muleApplicationModelLoader.getRuntimeVersion(),
                                                    muleApplicationModelLoader.getRuntimeProduct());

    return of(targetRuntime,
              new ApplicationRunConfigurationFactory(getLog(), munitTest, munitTags, skipAfterFailure, targetRuntime,
                                                     workingDirectoryGenerator, munitTestsDirectory, coverage, pluginVersion,
                                                     project, session).create());
  }

  @Override
  protected ResultPrinterFactory getResultPrinterFactory() {
    return new ApplicationResultPrinterFactory(getLog())
        .withCoverageSummaryReport(coverage, muleApplicationModelLoader.getRuntimeProduct())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory);
  }

  @Override
  protected ApplicationStructureGenerator getApplicationStructureGenerator() {
    return new MuleApplicationStructureGenerator(project.getBasedir().toPath(), Paths.get(project.getBuild().getDirectory()),
                                                 new ResolvedPom(project), createDependencies());
  }

  @Override
  protected File getMuleApplicationJsonPath() {
    Path projectBuildDirectoryPath = Paths.get(project.getBuild().getDirectory());
    Path muleApplicationJsonPath = projectBuildDirectoryPath
        .resolve(META_INF.value()).resolve(MULE_ARTIFACT.value()).resolve(MULE_ARTIFACT_JSON_FILE_NAME);

    return muleApplicationJsonPath.toFile();
  }

  private void saveRunDataToFile(RunResult runResult) {
    saveAsJsonDataToFile(coverage == null ? new Coverage() : coverage, coverageConfigDataFile);
    runResult.getApplicationCoverageReport().ifPresent(this::saveCoverageReportDataToFile);
  }

  private void saveCoverageReportDataToFile(ApplicationCoverageReport applicationCoverageReport) {
    saveAsJsonDataToFile(applicationCoverageReport, coverageReportDataFile);
  }

  private List<BundleDependency> createDependencies() {
    Map<BundleDescriptor, Artifact> artifacts = project.getArtifacts().parallelStream()
        .collect(toMap(this::toDescriptor, identity(), (oldValue, newValue) -> {
          if (oldValue.getType().equals(newValue.getType())) {

            return oldValue.hasClassifier() ? oldValue : newValue;
          }

          throw new IllegalArgumentException("Duplicate keys with different values: " + oldValue + " != " + newValue);
        }));

    Map<BundleDescriptor, BundleScope> scopes = artifacts.entrySet().parallelStream()
        .collect(toMap(Map.Entry::getKey, entry -> toScope(entry.getValue())));

    Map<BundleDescriptor, URI> uris = artifacts.entrySet().parallelStream()
        .collect(toMap(Map.Entry::getKey, entry -> toUri(entry.getValue())));

    Set<BundleDescriptor> direct = new HashSet<>();
    Map<BundleDescriptor, Set<BundleDescriptor>> transitive = artifacts.keySet().parallelStream()
        .collect(toMap(identity(), unused -> new HashSet<>()));

    for (BundleDescriptor descriptor : artifacts.keySet()) {
      BundleDescriptor predecessor = toDescriptor(artifacts.get(descriptor).getDependencyTrail());
      transitive.getOrDefault(predecessor, direct).add(descriptor);
    }

    return direct.parallelStream()
        .map(descriptor -> toDependency(descriptor, scopes, uris, transitive))
        .collect(toList());
  }

  private BundleDescriptor toDescriptor(Artifact artifact) {
    return new BundleDescriptor.Builder()
        .setGroupId(artifact.getGroupId())
        .setArtifactId(artifact.getArtifactId())
        .setVersion(artifact.getVersion())
        .setType(artifact.getType())
        .setClassifier(artifact.getClassifier())
        .build();
  }

  private BundleDescriptor toDescriptor(List<String> trail) {
    return toDescriptor(trail.get(trail.size() - 2));
  }

  private BundleDescriptor toDescriptor(String id) {
    String[] parts = id.split(":");

    return new BundleDescriptor.Builder()
        .setGroupId(parts[0])
        .setArtifactId(parts[1])
        .setType(parts[2]) // Does not affect result (see BundleDescriptor#equals(Object))
        .setVersion(parts[3])
        .build();
  }

  private BundleScope toScope(Artifact artifact) {
    return Optional.ofNullable(artifact.getScope())
        .map(scope -> BundleScope.valueOf(scope.toUpperCase()))
        .orElse(BundleScope.COMPILE);
  }

  private URI toUri(Artifact artifact) {
    return artifact.getFile().toURI();
  }

  private BundleDependency toDependency(BundleDescriptor descriptor,
                                        Map<BundleDescriptor, BundleScope> scopes,
                                        Map<BundleDescriptor, URI> uris,
                                        Map<BundleDescriptor, Set<BundleDescriptor>> transitive) {
    BundleDependency.Builder builder = new BundleDependency.Builder();

    builder.setDescriptor(descriptor);
    builder.setScope(scopes.get(descriptor));
    builder.setBundleUri(uris.get(descriptor));

    for (BundleDescriptor transitiveDescriptor : transitive.get(descriptor)) {
      builder.addTransitiveDependency(toDependency(transitiveDescriptor, scopes, uris, transitive));
    }

    return builder.build();
  }
}
