/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import org.mule.munit.Coverage;
import org.mule.munit.plugin.maven.runner.model.Debugger;
import org.mule.munit.coverage.CoverageAuthorizator;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.plugin.maven.runtime.TargetRuntime;
import org.mule.munit.plugin.maven.util.BaseApplicationRunConfigurationFactory;
import org.mule.munit.remote.api.configuration.ContainerConfiguration;
import org.mule.munit.remote.api.configuration.CoverageConfiguration;
import org.mule.munit.remote.api.configuration.DebuggerConfiguration;
import org.mule.munit.remote.api.configuration.EmbeddedContainerConfiguration.EmbeddedContainerConfigurationBuilder;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.ServerPluginConfiguration;
import org.mule.munit.remote.api.configuration.ServerPluginConfiguration.ServerPluginConfigurationBuilder;

import java.io.File;
import java.util.Optional;
import java.util.Set;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;

/**
 * Creates {@link RunConfiguration} for an Application
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ApplicationRunConfigurationFactory extends BaseApplicationRunConfigurationFactory {

  private final Coverage coverage;
  private final String pluginVersion;
  private final boolean shouldRunCoverage;
  private final Debugger debugger;

  public ApplicationRunConfigurationFactory(Log log, String munitTest, String munitTags, Boolean skipAfterFailure,
                                            TargetRuntime targetRuntime,
                                            WorkingDirectoryGenerator workingDirectoryGenerator,
                                            File munitSrcFolder, Coverage coverage, Debugger debugger, String pluginVersion,
                                            MavenProject project,
                                            MavenSession session) {

    super(log, project.getArtifactId(), munitTest, munitTags, skipAfterFailure, targetRuntime, workingDirectoryGenerator,
          munitSrcFolder, project, session);
    checkArgument(isNotBlank(pluginVersion), "The pluginVersion must not be null nor empty");
    this.coverage = coverage;
    this.pluginVersion = pluginVersion;
    this.shouldRunCoverage = new CoverageAuthorizator(coverage, targetRuntime.getRuntimeProduct()).shouldRunCoverage();
    this.debugger = Optional.ofNullable(debugger).orElseGet(() -> Debugger.fromString(null));
  }

  @Override
  protected RunConfiguration.RunConfigurationBuilder getRunConfigurationBuilder() throws MojoExecutionException {
    RunConfiguration.RunConfigurationBuilder builder = super.getRunConfigurationBuilder();
    return builder
        .withCoverageConfiguration(buildCoverageConfiguration(shouldRunCoverage))
        .withDebuggerConfiguration(buildDebuggerConfiguration());
  }

  @Override
  protected ContainerConfiguration.ContainerConfigurationBuilder getContainerConfigurationBuilder() {
    EmbeddedContainerConfigurationBuilder builder =
        (EmbeddedContainerConfigurationBuilder) super.getContainerConfigurationBuilder();

    if (shouldRunCoverage) {
      ServerPluginConfiguration coveragePluginConfiguration = getCoverageServerPluginConfiguration(pluginVersion);
      builder.withServerPluginConfiguration(coveragePluginConfiguration);
    }

    if (debugger.getAddDebugger()) {
      builder.withServerPluginConfiguration(getDebuggerPluginConfiguration(debugger.getDebuggerVersion()));
    }

    return builder;
  }

  private CoverageConfiguration buildCoverageConfiguration(Boolean shouldRunCoverage) {
    Set<String> coverageIgnoreFlows = coverage != null ? coverage.getIgnoreFlows() : emptySet();
    Set<String> coverageIgnoreFiles = coverage != null ? coverage.getIgnoreFiles() : emptySet();

    CoverageConfiguration coverageConfiguration = new CoverageConfiguration.CoverageConfigurationBuilder()
        .withShouldRunCoverage(shouldRunCoverage)
        .withSuitePaths(locateMunitTestSuitesToRun())
        .withIgnoredFiles(coverageIgnoreFiles)
        .withIgnoredFlowNames(coverageIgnoreFlows)
        .build();

    return coverageConfiguration;
  }

  private DebuggerConfiguration buildDebuggerConfiguration() {
    if (!debugger.getAddDebugger()) {
      return null;
    }

    return new DebuggerConfiguration(debugger.getDebuggerPort(), debugger.getLockId());
  }

  private ServerPluginConfiguration getCoverageServerPluginConfiguration(String coveragePluginVersion) {
    return new ServerPluginConfigurationBuilder()
        .withGroupId("com.mulesoft.munit.plugins")
        .withArtifactId("munit-coverage-plugin")
        .withVersion(coveragePluginVersion)
        .withClassifier("mule-server-plugin")
        .withType("jar")
        .build();
  }

  private ServerPluginConfiguration getDebuggerPluginConfiguration(String debuggerPluginVersion) {
    return new ServerPluginConfigurationBuilder()
        .withGroupId("com.mulesoft")
        .withArtifactId("mule-dx-mule-debugger")
        .withVersion(debuggerPluginVersion)
        .withType("jar")
        .withClassifier("mule-server-plugin")
        .build();
  }

}
