/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static jersey.repackaged.com.google.common.collect.ImmutableMap.of;
import static org.mule.munit.plugin.maven.runtime.RuntimeProducts.EE;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.runtime.api.deployment.meta.Product.MULE_EE;

import java.io.File;
import java.net.URI;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;
import org.mule.munit.coverage.CoverageLimitsChecker;
import org.mule.munit.mojo.ApplicationResultPrinterFactory;
import org.mule.munit.mojo.exceptions.MojoExecutionExceptionFactory;
import org.mule.munit.plugin.maven.AbstractMunitMojo;
import org.mule.munit.plugin.maven.runner.model.RunResult;
import org.mule.munit.plugin.maven.runtime.TargetRuntime;
import org.mule.munit.plugin.maven.util.ResultPrinterFactory;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.project.ApplicationStructureGenerator;
import org.mule.munit.remote.api.project.MuleApplicationStructureGenerator;
import org.mule.munit.remote.tools.client.BAT.BATClient;
import org.mule.munit.remote.tools.client.BAT.BATClientBase;
import org.mule.munit.remote.tools.client.BAT.model.request.TestType;
import org.mule.munit.util.ApplicationRunConfigurationFactory;
import org.mule.munit.util.CloudHubRunConfigurationFactory;
import org.mule.tools.api.classloader.model.SharedLibraryDependency;
import org.mule.tools.api.packager.builder.MulePackageBuilder;
import org.mule.tools.api.util.MavenComponents;
import org.mule.tools.client.authentication.AuthenticationServiceClient;
import org.mule.tools.client.authentication.model.Credentials;
import org.mule.tools.model.Deployment;
import org.mule.tools.model.anypoint.CloudHubDeployment;


/**
 * <p>
 * MUnit Mojo to run tests
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
@Mojo(name = "test", defaultPhase = LifecyclePhase.TEST, requiresDependencyResolution = ResolutionScope.COMPILE)
public class MUnitMojo extends AbstractMunitMojo {

  protected static final String GROUP_ID_ELEMENT = "groupId";
  protected static final String ARTIFACT_ID_ELEMENT = "artifactId";

  protected static final String MULE_MAVEN_PLUGIN_KEY = "org.mule.tools.maven:mule-maven-plugin";
  protected static final String SHARED_LIBRARIES_KEY = "sharedLibraries";

  @Parameter(property = "munit.coverage")
  protected Coverage coverage;

  @Parameter(defaultValue = "${munit.randomFailMessages}")
  private boolean randomFailMessages = false;

  @Parameter(defaultValue = "${plugin.version}")
  protected String pluginVersion;

  @Parameter(property = "munit.coverageReportData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-report.data")
  protected File coverageReportDataFile;

  @Parameter(property = "munit.coverageConfigData",
      defaultValue = "${project.build.directory}/munit-reports/coverage-config.data")
  protected File coverageConfigDataFile;

  protected MojoExecutionExceptionFactory exceptionFactory;
  protected CoverageLimitsChecker coverageLimitsChecker;

  @Override
  protected void init() throws MojoExecutionException {
    super.init();
    this.exceptionFactory = new MojoExecutionExceptionFactory(randomFailMessages);
    this.coverageLimitsChecker = new CoverageLimitsChecker(coverage, muleApplicationModelLoader.getRuntimeProduct(), getLog());
  }

  @Override
  protected void handleRunResult(TargetRuntime targetRuntime, RunResult runResult) {
    super.handleRunResult(targetRuntime, runResult);
    runResult.getApplicationCoverageReport().ifPresent(coverageLimitsChecker::setCoverageReport);
    saveRunDataToFile(runResult);
  }

  @Override
  protected void failBuildIfNecessary(Map<TargetRuntime, RunResult> runResults) throws MojoExecutionException {
    try {
      super.failBuildIfNecessary(runResults);

    } catch (MojoExecutionException cause) {
      throw exceptionFactory.buildException("Build Failed", cause);
    }

    if (coverage != null && coverage.isRunCoverage() && !EE.value().equals(muleApplicationModelLoader.getRuntimeProduct())) {
      getLog().warn("Coverage is a EE only feature and you've selected to run over CE");
    }

    if (coverageLimitsChecker.failBuild()) {
      throw new MojoExecutionException("Build Failed", new MojoFailureException("Coverage limits were not reached"));
    }
  }

  @Override
  protected Map<TargetRuntime, RunConfiguration> getRunConfigurations() throws MojoExecutionException {
    Deployment deployment = getDeployments().stream().filter(Objects::nonNull).findFirst().orElse(null);

    if (deployment != null && deployment instanceof CloudHubDeployment) {

      CloudHubDeployment cloudHubDeployment = (CloudHubDeployment) deployment;

      TargetRuntime targetRuntime =
          new TargetRuntime(deployment.getMuleVersion().orElse(muleApplicationModelLoader.getRuntimeVersion()), MULE_EE.name());
      return of(targetRuntime,
                new CloudHubRunConfigurationFactory(getLog(), munitTest, munitTags, skipAfterFailure, targetRuntime,
                                                    workingDirectoryGenerator, munitTestsDirectory, coverage, pluginVersion,
                                                    project, session, deployment, systemPropertyVariables, environmentVariables,
                                                    getExecutionId(cloudHubDeployment), clearParameters)
                                                        .create());
    } else {
      return super.getRunConfigurations();
    }
  }

  @Override
  protected RunConfiguration createRunConfiguration(TargetRuntime targetRuntime) throws MojoExecutionException {
    return new ApplicationRunConfigurationFactory(getLog(), munitTest, munitTags, skipAfterFailure, targetRuntime,
                                                  workingDirectoryGenerator, munitTestsDirectory, coverage, debugger,
                                                  pluginVersion,
                                                  project, session, clearParameters).create();
  }

  @Override
  protected ResultPrinterFactory getResultPrinterFactory() {
    return new ApplicationResultPrinterFactory(getLog())
        .withCoverageSummaryReport(coverage, muleApplicationModelLoader.getRuntimeProduct())
        .withSurefireReports(enableSurefireReports, surefireReportsFolder, effectiveSystemProperties)
        .withSonarReports(enableSonarReports, sonarReportsFolder)
        .withTestOutputReports(redirectTestOutputToFile, testOutputDirectory);
  }

  @Override
  protected ApplicationStructureGenerator getApplicationStructureGenerator() {
    return new MuleApplicationStructureGenerator(projectBaseFolder.toPath(), outputDirectory.toPath())
        .withPom(new ResolvedPom(project))
        .withDependencies(createDependencies())
        .withPackageBuilder(new MulePackageBuilder())
        .withMavenComponents(getMavenComponents())
        .isHeavyWeight(cloudHubDeployment != null);
  }

  @Override
  protected File getMuleApplicationJsonPath() {
    Path projectBuildDirectoryPath = Paths.get(project.getBuild().getDirectory());
    Path muleApplicationJsonPath = projectBuildDirectoryPath
        .resolve(META_INF.value()).resolve(MULE_ARTIFACT.value()).resolve(MULE_ARTIFACT_JSON_FILE_NAME);

    return muleApplicationJsonPath.toFile();
  }

  private void saveRunDataToFile(RunResult runResult) {
    saveAsJsonDataToFile(coverage == null ? new Coverage() : coverage, coverageConfigDataFile);
    runResult.getApplicationCoverageReport().ifPresent(this::saveCoverageReportDataToFile);
  }

  private void saveCoverageReportDataToFile(ApplicationCoverageReport applicationCoverageReport) {
    saveAsJsonDataToFile(applicationCoverageReport, coverageReportDataFile);
  }

  private List<BundleDependency> createDependencies() {
    Set<Artifact> artifacts = project.getArtifacts();

    Set<Artifact> direct = new TreeSet<>();
    Map<Artifact, Set<Artifact>> transitive = artifacts.parallelStream()
        .collect(toMap(identity(), unused -> new TreeSet<>()));

    for (Artifact artifact : artifacts) {
      Artifact predecessor = findArtifact(artifacts, artifact.getDependencyTrail());
      transitive.getOrDefault(predecessor, direct).add(artifact);
    }

    return direct.parallelStream()
        .map(artifact -> toDependency(artifact, transitive))
        .collect(toList());
  }

  private BundleDescriptor toDescriptor(Artifact artifact) {
    return new BundleDescriptor.Builder()
        .setGroupId(artifact.getGroupId())
        .setArtifactId(artifact.getArtifactId())
        .setVersion(artifact.getVersion())
        .setType(artifact.getType())
        .setClassifier(artifact.getClassifier())
        .build();
  }

  private Artifact findArtifact(Set<Artifact> artifacts, List<String> trail) {
    return findArtifact(artifacts, trail.get(trail.size() - 2));
  }

  private Artifact findArtifact(Set<Artifact> artifacts, String id) {
    return artifacts.parallelStream()
        .filter(artifact -> Objects.equals(artifact.getId(), id))
        .findAny().orElse(null);
  }

  private BundleScope toScope(Artifact artifact) {
    return Optional.ofNullable(artifact.getScope())
        .map(scope -> BundleScope.valueOf(scope.toUpperCase()))
        .orElse(BundleScope.COMPILE);
  }

  private URI toUri(Artifact artifact) {
    return artifact.getFile().toURI();
  }

  private BundleDependency toDependency(Artifact artifact, Map<Artifact, Set<Artifact>> transitive) {
    BundleDependency.Builder builder = new BundleDependency.Builder();

    builder.setDescriptor(toDescriptor(artifact));
    builder.setScope(toScope(artifact));
    builder.setBundleUri(toUri(artifact));

    for (Artifact transitiveArtifact : transitive.get(artifact)) {
      builder.addTransitiveDependency(toDependency(transitiveArtifact, transitive));
    }

    return builder.build();
  }

  private String getChildParameterValue(Xpp3Dom element, String childName, boolean validate) {
    Xpp3Dom child = element.getChild(childName);
    String childValue = child != null ? child.getValue() : null;
    if (StringUtils.isEmpty(childValue) && validate) {
      throw new IllegalArgumentException("Expecting child element with not null value " + childName);
    }
    return childValue;
  }

  private MavenComponents getMavenComponents() {
    return new MavenComponents()
        .withLog(getLog())
        .withProject(project)
        .withOutputDirectory(outputDirectory)
        .withSession(session)
        .withSharedLibraries(getSharedLibraries())
        .withProjectBuilder(projectBuilder)
        .withRepositorySystem(mavenRepositorySystem)
        .withLocalRepository(localRepository)
        .withRemoteArtifactRepositories(remoteArtifactRepositories)
        .withAdditionalPluginDependencies(getAdditionalPluginDependencies())
        .withProjectBaseFolder(projectBaseFolder);
  }

  private List<org.mule.tools.api.classloader.model.resolver.Plugin> getAdditionalPluginDependencies() {

    List<org.mule.tools.api.classloader.model.resolver.Plugin> plugins = new Vector<>();

    try {
      Plugin plugin = project.getPlugin(MULE_MAVEN_PLUGIN_KEY);

      if (plugin != null) {
        Xpp3Dom configuration = (Xpp3Dom) plugin.getConfiguration();

        if (configuration != null) {
          Xpp3Dom additionalPluginElement = configuration.getChild("additionalPluginDependencies");

          if (additionalPluginElement != null) {
            Xpp3Dom libraries[] = additionalPluginElement.getChildren();

            Arrays.stream(libraries)
                .forEach(library -> {
                  Xpp3Dom additionalPluginDepedencies = configuration.getChild("additionalPluginDependencies");
                  if (additionalPluginDepedencies != null) {
                    Xpp3Dom pluginDependency =
                        additionalPluginDepedencies.getChild("plugin");
                    String pluginGroupId = getChildParameterValue(pluginDependency, GROUP_ID_ELEMENT, true);
                    String pluginArtifactId = getChildParameterValue(pluginDependency, ARTIFACT_ID_ELEMENT, true);

                    org.mule.tools.api.classloader.model.resolver.Plugin dependency =
                        new org.mule.tools.api.classloader.model.resolver.Plugin();
                    dependency.setArtifactId(pluginArtifactId);
                    dependency.setGroupId(pluginGroupId);

                    List<Dependency> additionalDependencies = new ArrayList<>();
                    Arrays.stream(pluginDependency.getChild("additionalDependencies").getChildren())
                        .forEach(lib -> {
                          Dependency additionalDependency = new Dependency();
                          String additionalDependencyGroupId = getChildParameterValue(lib, GROUP_ID_ELEMENT, true);
                          String additionalDependencyArtifactId = getChildParameterValue(lib, ARTIFACT_ID_ELEMENT, true);
                          String version = getChildParameterValue(lib, "version", true);
                          additionalDependency.setArtifactId(additionalDependencyArtifactId);
                          additionalDependency.setGroupId(additionalDependencyGroupId);
                          additionalDependency.setVersion(version);
                          additionalDependencies.add(additionalDependency);
                        });

                    dependency.setAdditionalDependencies(additionalDependencies);
                    plugins.add(dependency);
                  }
                });
          }
        }
      }

    } catch (Exception e) {
      getLog().warn("Unable to get sharedLibraries from mule-maven-plugin");
    }

    return plugins;
  }

  private List<SharedLibraryDependency> getSharedLibraries() {

    List<SharedLibraryDependency> sharedLibraries = new Vector<>();

    try {
      Plugin plugin = project.getPlugin(MULE_MAVEN_PLUGIN_KEY);

      if (plugin != null) {
        Xpp3Dom configuration = (Xpp3Dom) plugin.getConfiguration();

        if (configuration != null) {
          Xpp3Dom sharedLibrariesElement = configuration.getChild(SHARED_LIBRARIES_KEY);

          if (sharedLibrariesElement != null) {
            Xpp3Dom libraries[] = sharedLibrariesElement.getChildren();

            Arrays.stream(libraries)
                .forEach(library -> {
                  String pluginGroupId = getChildParameterValue(library, GROUP_ID_ELEMENT, true);
                  String pluginArtifactId =
                      getChildParameterValue(library, ARTIFACT_ID_ELEMENT, true);

                  SharedLibraryDependency sharedLib = new SharedLibraryDependency();
                  sharedLib.setArtifactId(pluginArtifactId);
                  sharedLib.setGroupId(pluginGroupId);
                  sharedLibraries.add(sharedLib);
                });
          }
        }
      }

    } catch (Exception e) {
      getLog().warn("Unable to get sharedLibraries from mule-maven-plugin");
    }

    return sharedLibraries;
  }

  protected BATClient getBATClient(CloudHubDeployment deployment) {
    return new BATClient(deployment.getUri(), new Credentials(deployment.getUsername(), deployment.getPassword()),
                         new AuthenticationServiceClient(deployment.getUri(), true));
  }

  private String getExecutionId(CloudHubDeployment deployment) throws MojoExecutionException {
    try {
      BATClientBase client = getBATClient(deployment);
      return client.createExecution(TestType.MUNIT).getExecutionId();
    } catch (Exception ex) {
      throw new MojoExecutionException("An error occurred while trying to create the Execution.");
    }
  }
}
