/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.io.File;
import java.util.Map;
import java.util.Set;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.mule.munit.Coverage;
import org.mule.munit.coverage.CoverageAuthorizator;
import org.mule.munit.plugin.maven.runner.structure.WorkingDirectoryGenerator;
import org.mule.munit.plugin.maven.util.BaseCloudHubRunConfigurationFactory;
import org.mule.munit.plugin.maven.runtime.TargetRuntime;
import org.mule.munit.remote.api.configuration.CoverageConfiguration;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.tools.model.Deployment;

/**
 * Creates {@link RunConfiguration} for CloudHub
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class CloudHubRunConfigurationFactory extends BaseCloudHubRunConfigurationFactory {

  private Coverage coverage;
  private boolean shouldRunCoverage;

  public CloudHubRunConfigurationFactory(Log log, String munitTest, String munitTags, Boolean skipAfterFailure,
                                         TargetRuntime targetRuntime,
                                         WorkingDirectoryGenerator workingDirectoryGenerator,
                                         File munitSrcFolder, Coverage coverage, String pluginVersion,
                                         MavenProject project,
                                         MavenSession session, Deployment deployment, Map<String, String> systemPropertyVariables,
                                         Map<String, String> environmentVariables, String executionId, boolean clearParameters) {

    super(log, project.getArtifactId(), munitTest, munitTags, skipAfterFailure, targetRuntime, workingDirectoryGenerator,
          munitSrcFolder, project, session, deployment, systemPropertyVariables, environmentVariables, executionId,
          clearParameters);
    checkArgument(isNotBlank(pluginVersion), "The pluginVersion must not be null nor empty");

    this.coverage = coverage;
    this.shouldRunCoverage = new CoverageAuthorizator(coverage, targetRuntime.getRuntimeProduct()).shouldRunCoverage();
  }

  @Override
  protected RunConfiguration.RunConfigurationBuilder getRunConfigurationBuilder() throws MojoExecutionException {
    RunConfiguration.RunConfigurationBuilder builder = super.getRunConfigurationBuilder();
    return builder.withCoverageConfiguration(buildCoverageConfiguration(shouldRunCoverage));
  }

  private CoverageConfiguration buildCoverageConfiguration(Boolean shouldRunCoverage) {
    Set<String> coverageIgnoreFlows = coverage != null ? coverage.getIgnoreFlows() : emptySet();
    Set<String> coverageIgnoreFiles = coverage != null ? coverage.getIgnoreFiles() : emptySet();

    CoverageConfiguration coverageConfiguration = new CoverageConfiguration.CoverageConfigurationBuilder()
        .withShouldRunCoverage(shouldRunCoverage)
        .withSuitePaths(locateMunitTestSuitesToRun())
        .withIgnoredFiles(coverageIgnoreFiles)
        .withIgnoredFlowNames(coverageIgnoreFlows)
        .build();

    return coverageConfiguration;
  }

}
