/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.List;

import org.apache.maven.project.MavenProject;

import org.mule.tools.api.classloader.model.ArtifactCoordinates;
import org.mule.tools.api.packager.Pom;


/**
 * <p>
 * A {@link Pom} created from a {@link MavenProject}.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.2.1
 */
class ResolvedPom implements Pom {

  private final String groupId, artifactId, version;
  private final List<ArtifactCoordinates> dependencies;
  private final List<Path> resourcesLocation = new LinkedList<>();

  ResolvedPom(MavenProject project) {
    this.groupId = project.getGroupId();
    this.artifactId = project.getArtifactId();
    this.version = project.getVersion();

    this.dependencies = project.getDependencies().parallelStream()
        .map(dependency -> new ArtifactCoordinates(dependency.getGroupId(), dependency.getArtifactId(),
                                                   dependency.getVersion(), dependency.getType(),
                                                   dependency.getClassifier(), dependency.getScope()))
        .collect(toList());

    project.getResources().forEach(resource -> this.resourcesLocation.add(Paths.get(resource.getDirectory())));
  }

  @Override
  public void persist(Path pom) {}

  @Override
  public String getGroupId() {
    return groupId;
  }

  @Override
  public String getArtifactId() {
    return artifactId;
  }

  @Override
  public String getVersion() {
    return version;
  }

  @Override
  public List<ArtifactCoordinates> getDependencies() {
    return unmodifiableList(dependencies);
  }

  @Override
  public List<Path> getResourcesLocation() {
    return unmodifiableList(resourcesLocation);
  }
}
