/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.api.annotations.Category;
import org.mule.api.annotations.Configurable;
import org.mule.api.annotations.Module;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.display.Password;
import org.mule.api.annotations.display.Placement;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.api.lifecycle.InitialisationException;

import javax.annotation.PostConstruct;

/**
 * <p>FTP server for Integration tests.</p>
 * <p/>
 * <p>With this module you can start a FTP server on your local machine </p>
 *
 * @author Mulesoft Inc.
 */
@Module(name = "ftpserver", schemaVersion = "1.0", minMuleVersion = "3.4.0", friendlyName = "FTP Server")
@Category(name = "org.mule.tooling.category.munit.utils", description = "Munit tools")
public class FTPServerModule {
    private static transient Logger logger = LogManager.getLogger(FTPServerModule.class);

    /**
     * <p>FTP/SFTP server port.</p>
     * <p>If is not defined, the default value is 22.</p>
     */
    @Configurable
    @Optional
    @Default("22")
    @Placement(tab="General", group = "General", order = 1)
    private int port;

    /**
     * <p>Defines if is FTP over ssh(SFTP).</p>
     * <p>If is not defined,the default value is false.</p>
     */
    @Configurable
    @Optional
    @Default("false")
    private boolean secure;

    /**
     * <p>Defines if the FTP anonymous user account is enabled.</p>
     * <p>If is not defined,the default value is true.</p>
     */
    @Configurable
    @Optional
    @Default("true")
    private boolean anonymous;

    /**
     * <p>FTP/SFTP username.</p>
     * <p>If is not defined, the default value is empty.</p>
     */
    @Configurable
    @Optional
    @Default("")
    @Placement(tab="General", group = "General", order = 2)
    private String username;

    /**
     * <p>FTP/SFTP password.</p>
     * <p>If is not defined, the default value is empty.</p>
     */
    @Configurable
    @Optional
    @Password
    @Default("")
    @Placement(tab="General", group = "General", order = 3)
    private String password;

    /**
     * <p>FTP/SFTP home folder.</p>
     * <p>If is not defined, the default value is the root of the filesystem.</p>
     */
    @Configurable
    @Optional
    @Default("")
    @Placement(tab="General", group = "General", order = 4)
    private String homeDir;

    private Server server;


    /**
     * <p>Starts the server</p>
     * <p/>
     * {@sample.xml ../../../doc/SFTPServer-connector.xml.sample sftpserver:startServer}
     */
    @Processor
    public void startServer() {
        server.start();
    }

    /**
     * <p>check if a file exists.</p
     * <p/>
     * {@sample.xml ../../../doc/SFTPServer-connector.xml.sample sftpserver:containsFiles}
     *
     * @param file The file name that you want to check if is in the ftp.
     * @param path Relative path where it is going to be the file relatively to the home directory.
     */
    @Processor
    public void containsFiles(String file, String path) {
        server.containsFiles(file, getAbsolutePath(path));
    }

    private String getAbsolutePath(String path) {
        return StringUtils.isNotBlank(homeDir) ? homeDir +  path : path;
    }

    /**
     * <p>Stops the server</p>
     * <p/>
     * {@sample.xml ../../../doc/SFTPServer-connector.xml.sample sftpserver:stopServer}
     */
    @Processor
    public void stopServer() {
        try {
            server.stop();
        } catch (Throwable t) {
            logger.error("The server has failed to stop", t);
        }
    }


    /**
     * <p>Remove created files</p>
     * <p/>
     * {@sample.xml ../../../doc/SFTPServer-connector.xml.sample sftpserver:remove}
     *
     * @param path Path to be removed.
     */
    @Processor
    public void remove(String path) {
        server.remove(getAbsolutePath(path));
    }

    public void setPort(int port) {
        this.port = port;
    }

    public void setSecure(boolean secure) {
        this.secure = secure;
    }

    public void setAnonymous(boolean anonymous) { this.anonymous = anonymous; }

    public void setUsername(String username) {
        this.username = username;
    }

    public void setPassword(String password) { this.password = password; }

    public void setHomeDir(String homeDir) {
        if(homeDir != null && homeDir.startsWith("~")) {
            homeDir = System.getProperty("user.home") + StringUtils.substring(homeDir, 1);
        }
        this.homeDir = homeDir;
    }

    @PostConstruct
    public void buildServer() throws InitialisationException {
        if (secure) {
            server = new SFTPServer();
        } else {
            server = new FTPServer();
        }
        server.initialize(port, username, password, anonymous, getHomeDir());
    }

    public int getPort() {
        return port;
    }

    public boolean isSecure() {
        return secure;
    }

    public boolean isAnonymous() { return anonymous; }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public String getHomeDir() {
        return homeDir;
    }
}
