/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import org.apache.commons.lang.StringUtils;
import org.apache.ftpserver.ftplet.*;
import org.apache.ftpserver.usermanager.AnonymousAuthentication;
import org.apache.ftpserver.usermanager.UsernamePasswordAuthentication;
import org.apache.sshd.server.auth.password.PasswordAuthenticator;
import org.apache.sshd.server.session.ServerSession;

/**
 * <p>Mock user Manager for testing</p>
 *
 * @author Mulesoft Inc.
 */
public class MockUserManagerAuthenticator implements UserManager, PasswordAuthenticator {
    private static final String ADMIN_NAME = "mock";
    private static final String ANONYMOUS_USERNAME = "anonymous";
    private static final String EMPTY_STRING = "";
    private MockUser user;
    private MockUser anonymousUser = new MockUser(ANONYMOUS_USERNAME, EMPTY_STRING);
    private MockUser emptyUser = new MockUser(EMPTY_STRING,EMPTY_STRING);
    private boolean anonymous;

    public MockUserManagerAuthenticator(String username, String password, boolean anonymous, String homeDir) {
        this.anonymous  = anonymous;
        enforceMinimumConfigurationIsSetUp(username, password, anonymous);
        this.user = isUsernameOrPasswordNull(username, password) ? emptyUser : new MockUser(username, password);
        setHomeDirectoryToUsers(homeDir);
    }

    private void enforceMinimumConfigurationIsSetUp(String username, String password, boolean anonymous) {
        if(!anonymous && isUsernameOrPasswordNullOrEmpty(username, password)) {
            throw new RuntimeException("Username and password must be defined on the FTP server configuration");
        }
    }

    private boolean isUsernameOrPasswordNullOrEmpty(String username, String password) {
        return isUsernameOrPasswordNull(username,password) || isUsernameOrPasswordEmpty(username, password);
    }

    private boolean isUsernameOrPasswordEmpty(String username, String password) {
        return (StringUtils.isBlank(username) || password.equals(""));
    }

    private boolean isUsernameOrPasswordNull(String username, String password) {
        return (username == null) || (password == null);
    }

    public User getUserByName(String name) throws FtpException {
        if(StringUtils.equals(name, ANONYMOUS_USERNAME)) {
            return anonymousUser;
        }
        if(doesExist(name)) {
            return user;
        }
        return emptyUser;
    }

    public String[] getAllUserNames() throws FtpException {
        return new String[]{ user.getName() };
    }

    public void delete(String s) throws FtpException {

    }

    public void save(User user) throws FtpException {
    }

    public boolean doesExist(String name) throws FtpException {
        return StringUtils.equals(name, user.getName());
    }

    public User authenticate(Authentication authentication) throws AuthenticationFailedException {
        if(anonymous) {
            return getAnonymousOrEmptyUserBasedOnAuthentication(authentication);
        }
        else {
            return getUserBasedOnUsernamePasswordAuthentication(authentication);
        }
    }

    private User getUserBasedOnUsernamePasswordAuthentication(Authentication authentication) {
        if(!isUserTryingToLoginAsAnonymous(authentication) && isUserNameAndPasswordAuthentication(authentication)) {
            if (doesUsernameAndPasswordMatch(authentication)) {
                return user;
            }
        }
        return null;
    }

    private User getAnonymousOrEmptyUserBasedOnAuthentication(Authentication authentication) {
        return isUserTryingToLoginAsAnonymous(authentication) ? anonymousUser : emptyUser;
    }

    private boolean isUserTryingToLoginAsAnonymous(Authentication authentication) { // If the FTP client tries to connect using the username "anonymous",
        return authentication instanceof AnonymousAuthentication;                   // this is the type of authentication that is sent to the user manager.
    }

    private boolean doesUsernameAndPasswordMatch(Authentication authentication) {
        UsernamePasswordAuthentication usernamePasswordAuth = (UsernamePasswordAuthentication) authentication;
        return usernamePasswordAuth.getUsername().equals(user.getName()) &&
               usernamePasswordAuth.getPassword().equals(user.getPassword());

    }

    private boolean isUserNameAndPasswordAuthentication(Authentication authentication) { // However, if the FTP client tries to connect using any username that is
        return authentication instanceof UsernamePasswordAuthentication;                 // different from "anonymous", the type of authentication that is sent to the
    }                                                                                    // user manager is UsernamePasswordAuthentication.

    public String getAdminName() throws FtpException {
        return ADMIN_NAME;
    }

    public boolean isAdmin(String s) throws FtpException {
        return true;
    }

    public void setHomeDirectoryToUsers(String homeDir) {
        user.setHomeDirectory(homeDir);
        anonymousUser.setHomeDirectory(homeDir);
        emptyUser.setHomeDirectory(homeDir);
    }

    public boolean authenticate(String username, String password, ServerSession session) {
        return (anonymous || nameAndPasswordMatch(username, password));
    }

    private boolean nameAndPasswordMatch(String name, String password) {
        return (user.getName().equals(name) && user.getPassword().equals(password));
    }
}
