/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.munit.core;

import java.io.File;
import java.io.FilenameFilter;

import static org.junit.Assert.assertTrue;

import org.apache.commons.lang3.StringUtils;


/**
 * <p>
 * Implementation of the FTP server
 * </p>
 *
 * @author Mulesoft Inc.
 */
public abstract class AbstractFTPServer {

  public static final String HOME_FOLDER_WILDCARD = "~";

  protected String homeDir;

  public AbstractFTPServer(String homeDir) {
    if (homeDir != null && homeDir.startsWith(HOME_FOLDER_WILDCARD)) {
      this.homeDir = System.getProperty("user.home") + StringUtils.substring(homeDir, 1);
    } else {
      this.homeDir = homeDir;
    }
  }

  public abstract void start();

  public abstract void stop();

  public String getHomeDir() {
    return homeDir;
  }

  public void remove(String path) {
    new File(getAbsolutePath(path)).delete();
  }

  public void containsFiles(String file, String path) {
    String absolutePath = getAbsolutePath(path);
    File[] filteredFiles = containsFilteredFiles(file, absolutePath);
    if (filteredFiles == null) {
      throw new RuntimeException("The path to the file does not exist or an I/O error has occurred");
    }
    assertTrue("The file was not found in the folder", filteredFiles.length > 0);
  }

  private File[] containsFilteredFiles(final String fileName, String path) {
    return new File(path).listFiles(new FilenameFilter() {

      public boolean accept(File file, String name) {
        return name.startsWith(fileName);
      }
    });
  }

  private String getAbsolutePath(String path) {
    return StringUtils.isNotBlank(homeDir) ? homeDir + path : path;
  }

  protected String getAdaptedPath(String homeDir) {
    return (homeDir == null ? "/" : homeDir);
  }

  protected MockUserManagerAuthenticator getUserManager(String username, String password, boolean anonymous) {
    return new MockUserManagerAuthenticator(username, password, anonymous, getAdaptedPath(getHomeDir()));
  }
}
