/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.munit.extension;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.core.AbstractFTPServer;
import org.mule.munit.core.ftp.FTPServer;
import org.mule.munit.core.sftp.SFTPServer;
import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.Password;
import org.mule.runtime.extension.api.annotation.param.display.Placement;

/**
 * @author Mulesoft Inc.
 */
public class FTPServerConnectionProvider implements CachedConnectionProvider<AbstractFTPServer>, Lifecycle {

  /**
   * <p>
   * FTP/SFTP server port.
   * </p>
   * <p>
   * If is not defined, the default value is 22.
   * </p>
   */
  @Parameter
  @Optional(defaultValue = "22")
  @Placement(tab = "General", order = 1)
  private int port;

  /**
   * <p>
   * FTP/SFTP username.
   * </p>
   * <p>
   * If is not defined, the default value is empty.
   * </p>
   */
  @Parameter
  @Optional(defaultValue = StringUtils.EMPTY)
  @Placement(tab = "General", order = 2)
  private String username;

  /**
   * <p>
   * FTP/SFTP password.
   * </p>
   * <p>
   * If is not defined, the default value is empty.
   * </p>
   */
  @Parameter
  @Optional(defaultValue = StringUtils.EMPTY)
  @Password
  @Placement(tab = "General", order = 3)
  private String password;

  /**
   * <p>
   * FTP/SFTP home folder.
   * </p>
   * <p>
   * If is not defined, the default value is the root of the filesystem.
   * </p>
   */
  @Parameter
  @Optional(defaultValue = StringUtils.EMPTY)
  @Placement(tab = "General", order = 4)
  private String homeDir;

  /**
   * <p>
   * Defines if is FTP over ssh(SFTP).
   * </p>
   * <p>
   * If is not defined,the default value is false.
   * </p>
   */
  @Parameter
  @Optional(defaultValue = "false")
  @Placement(order = 5)
  private boolean secure;

  /**
   * <p>
   * Defines if the FTP anonymous user account is enabled.
   * </p>
   * <p>
   * If is not defined,the default value is true.
   * </p>
   */
  @Parameter
  @Optional(defaultValue = "true")
  @Placement(order = 6)
  private boolean anonymous;


  private AbstractFTPServer server;

  @Override
  public AbstractFTPServer connect() throws ConnectionException {
    return server;
  }

  @Override
  public void disconnect(AbstractFTPServer databaseServer) {

  }

  @Override
  public ConnectionValidationResult validate(AbstractFTPServer databaseServer) {
    return ConnectionValidationResult.success();
  }

  @Override
  public void start() throws MuleException {
    if (secure) {
      server = new SFTPServer(port, username, password, anonymous, homeDir);
    } else {
      server = new FTPServer(port, username, password, anonymous, homeDir);
    }

    server.start();
  }

  @Override
  public void stop() throws MuleException {
    if (server != null) {
      server.stop();
    }
  }

  @Override
  public void dispose() {

  }

  @Override
  public void initialise() throws InitialisationException {

  }
}
