/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule;

import com.mulesoft.module.batch.api.notification.BatchNotification;
import com.mulesoft.module.batch.api.notification.BatchNotificationListener;
import org.mule.api.context.notification.ServerNotification;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class BatchSynchronizeListener implements BatchNotificationListener, Synchronize {
    private AtomicBoolean receivedBatchNotification = new AtomicBoolean(false);
    private AtomicInteger inputPhaseCounter = new AtomicInteger(Integer.MAX_VALUE);
    private AtomicInteger loadPhaseCounter = new AtomicInteger(Integer.MAX_VALUE);
    private AtomicInteger jobPhaseCounter = new AtomicInteger(Integer.MAX_VALUE);
    private AtomicInteger onCompletePhaseCounter = new AtomicInteger(Integer.MAX_VALUE);

    @Override
    public synchronized void onNotification(ServerNotification serverNotification) {
        int action = serverNotification.getAction();
        if(action == BatchNotification.LOAD_PHASE_PROGRESS ||
                action == BatchNotification.STEP_RECORD_START ||
                action == BatchNotification.STEP_RECORD_FAILED ||
                action == BatchNotification.STEP_RECORD_END ||
                action == BatchNotification.STEP_COMMIT_START ||
                action == BatchNotification.STEP_COMMIT_END ||
                action == BatchNotification.STEP_RECORD_FAILED ||
                action == BatchNotification.PROGRESS_UPDATE) {
            return;
        }
        if (action == BatchNotification.INPUT_PHASE_BEGIN) {
            receivedBatchNotification.set(true);
            increment(inputPhaseCounter);
        } else if (action == BatchNotification.INPUT_PHASE_END ||
                action == BatchNotification.INPUT_PHASE_FAILED) {
            inputPhaseCounter.decrementAndGet();
        } else if (action == BatchNotification.LOAD_PHASE_BEGIN) {
            increment(loadPhaseCounter);
        } else if (action == BatchNotification.LOAD_PHASE_END ||
                action == BatchNotification.LOAD_PHASE_FAILED) {
            loadPhaseCounter.decrementAndGet();
        } else if (action == BatchNotification.JOB_PROCESS_RECORDS_BEGIN) {
            increment(jobPhaseCounter);
        } else if (action == BatchNotification.JOB_PROCESS_RECORDS_FAILED ||
                action == BatchNotification.JOB_CANCELLED ||
                action == BatchNotification.JOB_SUCCESSFUL ||
                action == BatchNotification.JOB_STOPPED) {
            jobPhaseCounter.decrementAndGet();
        } else if (action == BatchNotification.ON_COMPLETE_BEGIN) {
            increment(onCompletePhaseCounter);
        } else if (action == BatchNotification.ON_COMPLETE_END ||
                action == BatchNotification.ON_COMPLETE_FAILED) {
            onCompletePhaseCounter.decrementAndGet();
        }
    }

    private synchronized void increment(AtomicInteger counter) {
        if(counter.get() == Integer.MAX_VALUE) {
            counter.set(1);
        } else {
            counter.incrementAndGet();
        }
    }

    @Override
    public synchronized boolean readyToContinue() {
        return !receivedBatchNotification.get() || (inputPhaseCounter.get() <= 0 && loadPhaseCounter.get() <= 0 && jobPhaseCounter.get() <= 0 && onCompletePhaseCounter.get() <= 0);
    }
}
