/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */

package psft.pt8.joa;

import com.mulesoft.peoplesoft.exception.PeopleSoftException;
import org.apache.log4j.Logger;

import java.lang.reflect.Field;
import java.util.Map;
import java.util.Set;

/**
 * Collection of helper methods for dealing with PeopleSoft CI.
 */
public class PeopleSoftCIHelper {

    private static final Logger logger = Logger.getLogger(PeopleSoftCIHelper.class);

    private CI ci;

    public PeopleSoftCIHelper(Object ci) {
        if (ci == null) {
            throw new PeopleSoftException(PeopleSoftCIHelper.class.getName() + " requires a CI object. No null value allowed");
        } else if (!(ci instanceof CI)) {
            throw new PeopleSoftException(PeopleSoftCIHelper.class.getName() + " requires a CI object as input. Entered: " + ci.getClass().getName());
        }

        this.ci = (CI) ci;
    }

    /**
     * Set the config value as a boolean, access the component in interactive as opposed to deferred processing mode.
     * @param inInteractiveMode The config value to use.
     */
    public void setInteractiveMode(boolean inInteractiveMode) {
        try {
            ci.setProperty("InteractiveMode", Boolean.valueOf(inInteractiveMode));
        } catch (JOAException e) {
            throw new PeopleSoftException(e);
        }
    }

    /**
     * Set the config value as boolean, correction mode where effective dated data is involved.
     * @param inGetHistoryItems The config value to use.
     */
    public void setGetHistoryItems(boolean inGetHistoryItems) {
        try {
            ci.setProperty("GetHistoryItems", Boolean.valueOf(inGetHistoryItems));
        } catch (JOAException e) {
            throw new PeopleSoftException(e);
        }
    }

    /**
     * Set the config value as boolean, correction mode with ability to modify history data.
     * @param inEditHistoryItems The config value to use.
     */
    public void setEditHistoryItems(boolean inEditHistoryItems) {
        try {
            ci.setProperty("EditHistoryItems", Boolean.valueOf(inEditHistoryItems));
        } catch (JOAException e) {
            throw new PeopleSoftException(e);
        }
    }

    /**
     * Get set of authorized operations
     * @return The resulting Set; the values.
     */
    @SuppressWarnings("unchecked")
    public Set<String> getSupportedOperations() {
        try {
            Field authMethods = ci.getClass().getDeclaredField("m_setAuthorizedMethods");
            authMethods.setAccessible(true);
            return (Set<String>) authMethods.get(ci);
        } catch (NoSuchFieldException e) {
            throw new PeopleSoftException("Can not find attribute with authorized methods", e);
        } catch (SecurityException e) {
            throw new PeopleSoftException("Can not access attribute with authorized methods", e);
        } catch (IllegalArgumentException e) {
            throw new PeopleSoftException(e);
        } catch (IllegalAccessException e) {
            throw new PeopleSoftException(e);
        }
    }

    /**
     * Get the collection of CIPropertyInfo.
     * @return The value.
     */
    public PeopleSoftPICollectionHelper getPropertyInfoCollection() {
        try {
            PeopleSoftPICollectionHelper peopleSoftPropertyInfoCollectionHelper = new PeopleSoftPICollectionHelper(ci.getProperty("PropertyInfoCollection"));
            if (logger.isDebugEnabled()) {
                logger.debug(peopleSoftPropertyInfoCollectionHelper.toJson());
            }

            return peopleSoftPropertyInfoCollectionHelper;
        } catch (JOAException e) {
            throw new PeopleSoftException("Can not get PropertyInfoCollection for CI " + ci.getNamespaceName(), e);
        }
    }

    /**
     * Get the create key collection of CIPropertyInfo.
     * @return The value.
     */
    public PeopleSoftPICollectionHelper getCreateKeyInfoCollection() {
        try {
            PeopleSoftPICollectionHelper peopleSoftPropertyInfoCollectionHelper = new PeopleSoftPICollectionHelper(ci.getProperty("CreateKeyInfoCollection"));
            if (logger.isDebugEnabled()) {
                logger.debug(peopleSoftPropertyInfoCollectionHelper.toJson());
            }

            return peopleSoftPropertyInfoCollectionHelper;
        } catch (JOAException e) {
            throw new PeopleSoftException("Can not get CreateKeyInfoCollection for CI " + ci.getNamespaceName(), e);
        }
    }

    /**
     * Get the get key collection of CIPropertyInfo.
     * @return The value.
     */
    public PeopleSoftPICollectionHelper getGetKeyInfoCollection() {
        try {
            PeopleSoftPICollectionHelper peopleSoftPropertyInfoCollectionHelper = new PeopleSoftPICollectionHelper(ci.getProperty("GetKeyInfoCollection"));
            if (logger.isDebugEnabled()) {
                logger.debug(peopleSoftPropertyInfoCollectionHelper.toJson());
            }

            return peopleSoftPropertyInfoCollectionHelper;
        } catch (JOAException e) {
            throw new PeopleSoftException("Can not get GetKeyInfoCollection for CI " + ci.getNamespaceName(), e);
        }
    }

    /**
     * Get the find key collection of CIPropertyInfo.
     * @return The value.
     */
    public PeopleSoftPICollectionHelper getFindKeyInfoCollection() {
        try {
            PeopleSoftPICollectionHelper peopleSoftPropertyInfoCollectionHelper = new PeopleSoftPICollectionHelper(ci.getProperty("FindKeyInfoCollection"));
            if (logger.isDebugEnabled()) {
                logger.debug(peopleSoftPropertyInfoCollectionHelper.toJson());
            }

            return peopleSoftPropertyInfoCollectionHelper;
        } catch (JOAException e) {
            throw new PeopleSoftException("Can not get FindKeyInfoCollection for CI " + ci.getNamespaceName(), e);
        }
    }

    /**
     * Set the property.
     * @param name The config name.
     * @param value The value.
     */
    public void setProperty(String name, Object value) {
        try {
            ci.setProperty(name, value);
        } catch (JOAException e) {
            throw new PeopleSoftException("Can not set property " + name + " on CI " + ci.getNamespaceName(), e);
        }
    }

    /**
     * Get properties, Map key fields.
     * @return The value.
     */
    @SuppressWarnings("unchecked")
    public Map<String, Object> getProperties() {
        try {
            Field entitiesMap = ci.getClass().getDeclaredField("m_mapKeys");
            entitiesMap.setAccessible(true);
            return (Map<String, Object>) entitiesMap.get(ci);
        } catch (NoSuchFieldException e) {
            throw new PeopleSoftException("Can not find attribute with entities map", e);
        } catch (SecurityException e) {
            throw new PeopleSoftException("Can not access attribute with entities map", e);
        } catch (IllegalArgumentException e) {
            throw new PeopleSoftException(e);
        } catch (IllegalAccessException e) {
            throw new PeopleSoftException(e);
        }
    }

    /**
     * Get the specified property.
     * @param name The config name.
     * @return The property.
     */
    public Object getProperty(String name) {
        try {
            return ci.getProperty(name);
        } catch (JOAException e) {
            throw new PeopleSoftException(e);
        }
    }

    /**
     * Invokes the operation on the specified Method
     * @param method The method.
     * @return The response object.
     */
    public Object invokeMethod(String method) {
        try {
            Object invokeMethod = ci.invokeMethod(method, new Object[0]);
            if (logger.isDebugEnabled()) {
                logger.debug("InvokeMethod response " + invokeMethod);
            }
            return invokeMethod;
        } catch (JOAException e) {
            throw new PeopleSoftException(e);
        }
    }
}
