/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.classpath;

import static org.mule.runtime.container.api.ModuleRepository.createModuleRepository;

import static java.lang.String.format;

import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.container.internal.DefaultModuleRepository;
import org.mule.runtime.container.internal.ModuleDiscoverer;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URL;

import com.google.gson.ExclusionStrategy;
import com.google.gson.FieldAttributes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

public class ModuleRepositoryGenerator {

  static {
    URL.setURLStreamHandlerFactory(new ConfigurableStreamHandlerFactory("classpath", new ClasspathHandler()));
  }

  private static final String MODULE_REPOSITORY_JSON = "module-repository.json";

  public static void main(String[] args) throws IOException {
    new ModuleRepositoryGenerator().generate(new File(args[0]));
  }

  public void generate(File outputFolder) throws IOException {
    File temporaryFolder = new File(outputFolder, "moduleRepository");
    temporaryFolder.mkdir();

    ModuleRepository moduleRepository = createModuleRepository(this.getClass().getClassLoader(),
                                                               serviceInterface -> new File(temporaryFolder, serviceInterface),
                                                               (serviceInterface, serviceFile) -> {
                                                                 try {
                                                                   return new URL("classpath:" + (serviceFile.getPath()
                                                                       .substring(outputFolder.getAbsolutePath().length() + 1)));
                                                                 } catch (MalformedURLException e) {
                                                                   throw new IllegalStateException(format("Error creating temporary service provider file for '%s'",
                                                                                                          serviceInterface),
                                                                                                   e);
                                                                 }
                                                               });
    // force initialization
    moduleRepository.getModules();

    File outputFile = new File(outputFolder, MODULE_REPOSITORY_JSON);
    if (outputFile.exists()) {
      outputFile.delete();
    }
    outputFile.createNewFile();
    try (OutputStreamWriter osw = new OutputStreamWriter(new FileOutputStream(outputFile))) {
      osw.write(new GsonBuilder()
          .setExclusionStrategies(new ExclusionStrategy() {

            @Override
            public boolean shouldSkipField(FieldAttributes f) {
              return false;
            }

            @Override
            public boolean shouldSkipClass(Class<?> clazz) {
              return ModuleDiscoverer.class.isAssignableFrom(clazz);
            }

          })
          .create().toJson(moduleRepository));
    }
  }

  public ModuleRepository read() throws IOException {
    try (Reader reader =
        new BufferedReader(new InputStreamReader(this.getClass().getResourceAsStream("/" + MODULE_REPOSITORY_JSON)))) {
      return new Gson().fromJson(reader, DefaultModuleRepository.class);
    }
  }
}
