/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector.extension;

import static org.mule.runtime.container.api.ContainerClassLoaderProvider.createContainerClassLoader;

import static java.util.Optional.empty;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.core.api.extension.MuleExtensionModelProvider;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.extension.api.extension.XmlSdk1ExtensionModelProvider;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.MuleDeployableArtifactClassLoader;

import com.mulesoft.mule.runtime.bti.api.extension.BtiExtensionModelProvider;
import com.mulesoft.mule.runtime.core.api.extension.MuleEeExtensionModelProvider;
import com.mulesoft.mule.runtime.gw.autodiscovery.internal.AutodiscoveryExtensionModelProvider;
import com.mulesoft.mule.runtime.http.policy.api.extension.HttpPolicyEeExtensionModelProvider;
import com.mulesoft.mule.runtime.module.batch.api.extension.BatchExtensionModelProvider;
import com.mulesoft.mule.runtime.module.serialization.kryo.api.extension.KryoSerializerEeExtensionModelProvider;
import com.mulesoft.mule.runtime.tracking.api.extension.TrackingEeExtensionModelProvider;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

public class DefaultExtensionModelLoader implements AutoCloseable {

  private static final LazyValue<Set<ExtensionModel>> RUNTIME_EXTENSION_MODELS =
      new LazyValue<>(() -> getRuntimeExtensionModels());

  public static Set<ExtensionModel> getRuntimeExtensionModels() {
    List<Supplier<ExtensionModel>> runtimeExtensionModels = new ArrayList<>();

    runtimeExtensionModels.add(() -> MuleExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> MuleExtensionModelProvider.getTlsExtensionModel());
    runtimeExtensionModels.add(() -> XmlSdk1ExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> MuleEeExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> BatchExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> BtiExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> HttpPolicyEeExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> KryoSerializerEeExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> TrackingEeExtensionModelProvider.getExtensionModel());
    runtimeExtensionModels.add(() -> AutodiscoveryExtensionModelProvider.getExtensionModel());

    return runtimeExtensionModels
        .parallelStream()
        .map(Supplier::get)
        .collect(toSet());
  }

  private final ArtifactClassLoader containerClassLoader;
  private final LazyValue<ApplicationDescriptor> applicationDescriptor;
  private final LazyValue<MuleDeployableArtifactClassLoader> applicationClassLoader;
  private final DefaultExtensionModelService service;

  public DefaultExtensionModelLoader(File explodedAppDir, ClassLoader parentClassloader, ModuleRepository moduleRepository)
      throws IOException {
    this.containerClassLoader = createContainerClassLoader(moduleRepository, parentClassloader);
    MuleArtifactResourcesRegistry resourcesRegistry =
        new MuleArtifactResourcesRegistry(moduleRepository, containerClassLoader);

    this.applicationDescriptor = new LazyValue<>(() -> resourcesRegistry.getApplicationDescriptorFactory()
        .create(explodedAppDir, empty()));
    this.applicationClassLoader = new LazyValue<>(() -> resourcesRegistry.getApplicationClassLoaderFactory()
        .createApplicationClassLoader(this.applicationDescriptor.get(), explodedAppDir));
    this.service = new DefaultExtensionModelService(resourcesRegistry, RUNTIME_EXTENSION_MODELS.get());
  }

  public Set<ExtensionModel> obtainExtensionModels() {
    return concat(RUNTIME_EXTENSION_MODELS.get().stream(),
                  service.loadExtensionsData(applicationClassLoader.get().getArtifactPluginClassLoaders()).stream())
                      .collect(toSet());
  }

  @Override
  public void close() {
    this.applicationClassLoader.ifComputed(appCl -> {
      appCl.dispose();
      appCl.getArtifactPluginClassLoaders().forEach(ArtifactClassLoader::dispose);
    });
    this.containerClassLoader.dispose();
  }

  public ApplicationDescriptor getApplicationDescriptor() {
    return applicationDescriptor.get();
  }

  public ClassLoader getApplicationClassLoader() {
    return applicationClassLoader.get();
  }
}
