/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.ang.introspector.extension.application;

import static java.util.Objects.requireNonNull;
import static org.mule.runtime.deployment.model.api.builder.DeployableArtifactClassLoaderFactoryProvider.applicationClassLoaderFactory;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.builder.ApplicationClassLoaderBuilder;
import org.mule.runtime.deployment.model.api.builder.ApplicationClassLoaderBuilderFactory;
import org.mule.runtime.deployment.model.api.builder.RegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.api.plugin.resolver.PluginDependenciesResolver;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.MuleDeployableArtifactClassLoader;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

/**
 * Factory that knows how to create a {@link ArtifactClassLoader} for applications.
 */
public class ApplicationClassLoaderFactory {

  private final ArtifactClassLoader containerArtifactClassLoader;
  private final RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;
  private final PluginDependenciesResolver pluginDependenciesResolver;

  public ApplicationClassLoaderFactory(ArtifactClassLoader containerArtifactClassLoader,
                                       RegionPluginClassLoadersFactory regionPluginClassLoadersFactory,
                                       PluginDependenciesResolver pluginDependenciesResolver) {
    requireNonNull(containerArtifactClassLoader, "containerArtifactClassLoader cannot be null");
    requireNonNull(regionPluginClassLoadersFactory, "regionPluginClassLoadersFactory cannot be null");
    requireNonNull(pluginDependenciesResolver, "pluginDependenciesResolver cannot be null");

    this.containerArtifactClassLoader = containerArtifactClassLoader;
    this.regionPluginClassLoadersFactory = regionPluginClassLoadersFactory;
    this.pluginDependenciesResolver = pluginDependenciesResolver;
  }

  public MuleDeployableArtifactClassLoader createApplicationClassLoader(ApplicationDescriptor applicationDescriptor,
                                                                        File workingDirectory) {
    try {
      ApplicationClassLoaderBuilder builder =
          new ApplicationClassLoaderBuilderFactory(applicationClassLoaderFactory(name -> new File(workingDirectory, name)),
                                                   regionPluginClassLoadersFactory)
                                                       .createArtifactClassLoaderBuilder();

      Set<ArtifactPluginDescriptor> domainPluginDescriptors = new HashSet<>();
      boolean hasDomainDependency = applicationDescriptor.getDomainDescriptor().isPresent();
      if (hasDomainDependency) {
        // TODO EE-7984 Add support for domains
        throw new UnsupportedOperationException("Apps depending on domains are not yet supported");
      } else {
        builder.setDomainParentClassLoader(containerArtifactClassLoader);
      }

      pluginDependenciesResolver.resolve(domainPluginDescriptors, new ArrayList<>(applicationDescriptor.getPlugins()), true)
          .stream()
          .forEach(builder::addArtifactPluginDescriptors);

      builder.setArtifactDescriptor(applicationDescriptor);
      return builder.build();
    } catch (Exception e) {
      throw new MuleRuntimeException(e);
    }
  }
}
