/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.command;

import static picocli.CommandLine.ParameterException;

import java.io.PrintWriter;

import org.springframework.stereotype.Component;
import picocli.CommandLine;
import picocli.CommandLine.IParameterExceptionHandler;
import picocli.CommandLine.Model.CommandSpec;
import picocli.CommandLine.UnmatchedArgumentException;

/**
 * This class handles any exception which happened during parsing of parameters at the moment the tool was executed.
 */
@Component
public class ParameterExceptionHandler implements IParameterExceptionHandler {

  @Override
  public int handleParseException(ParameterException ex, String[] args) throws Exception {
    // @todo[DEL-2805]: Given that logging is getting more traction due to auditability needs, we might need to rethink this
    // implementation because this logging is not part of the default logging mechanism

    CommandLine cmd = ex.getCommandLine();
    PrintWriter err = cmd.getErr();

    // if tracing at DEBUG level, show the location of the issue
    if ("DEBUG".equalsIgnoreCase(System.getProperty("picocli.trace"))) {
      err.println(cmd.getColorScheme().stackTraceText(ex));
    }

    err.println(cmd.getColorScheme().errorText(ex.getMessage())); // bold red
    UnmatchedArgumentException.printSuggestions(ex, err);
    err.print(cmd.getHelp().fullSynopsis());

    CommandSpec spec = cmd.getCommandSpec();
    err.printf("Try '%s --help' for more information.%n", spec.qualifiedName());

    return cmd.getExitCodeExceptionMapper() != null
        ? cmd.getExitCodeExceptionMapper().getExitCode(ex)
        : spec.exitCodeOnInvalidInput();
  }
}
