/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.command;

import static picocli.CommandLine.Command;
import static picocli.CommandLine.Option;

import com.mulesoft.runtime.upgrade.tool.domain.UpgradeOptions;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeService;

import java.io.IOException;
import java.nio.file.Path;
import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * This class implements the upgrade command received by command line.
 */
@Component
@Command(name = "upgrade", description = "Upgrades a Mule Runtime to a newer version")
public class UpgradeCommand implements Callable<Integer> {

  private static final Logger LOGGER = LoggerFactory.getLogger(UpgradeCommand.class);

  @Option(names = {"-o", "--old-mule-distro"}, description = "File path of Mule Runtime distribution to upgrade", required = true)
  Path oldMuleDistro;

  @Option(names = {"-n", "--new-mule-distro"},
      description = "File path of the newer Mule Runtime distribution used for the upgrade", required = true)
  Path newMuleDistro;

  @Option(names = {"-f", "--force"}, description = "Perform all the destructive operations like overwriting the backup or " +
      "removing the applied patches without user confirmation")
  boolean force;

  @Option(names = {"-h", "--help"}, usageHelp = true, description = "display this help message")
  boolean usageHelpRequested;

  @Autowired
  private UpgradeService upgradeService;

  public Integer call() throws IOException {
    LOGGER.info("Initiating upgrade process... ");
    LOGGER.debug("Old Mule distribution location: [{}]. New Mule distribution location: [{}]", oldMuleDistro, newMuleDistro);
    UpgradeOptions upgradeOptions = UpgradeOptions.builder().force(force).build();
    upgradeService.upgrade(oldMuleDistro, newMuleDistro, upgradeOptions);
    return 0;
  }

  public void setOldMuleDistro(Path oldMuleDistro) {
    this.oldMuleDistro = oldMuleDistro;
  }

  public void setNewMuleDistro(Path newMuleDistro) {
    this.newMuleDistro = newMuleDistro;
  }
}
