/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor.builder;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatBackupDescriptor;
import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.service.api.AppService;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.PreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;
import com.mulesoft.runtime.upgrade.tool.service.api.YamlService;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A service for backup related operations.
 */
@Service
public class DefaultBackupService implements BackupService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultBackupService.class);

  @Autowired
  private AppService appService;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  @Autowired
  private YamlService yamlService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private PreconditionsValidatorService preconditionsValidatorService;

  // For managing the time in tests
  private Clock clock;

  /**
   * This function creates the backup descriptor.
   *
   * @param muleDistribution                a {@link MuleDistribution} representing the Mule distro where the backup will be taken
   *                                        from.
   * @param relativePathOfFilesToBeBackedUp the list of all relative paths to the Mule distribution that should be backed up.
   * @throws IOException
   */
  @Override
  public void create(MuleDistribution muleDistribution, List<Path> relativePathOfFilesToBeBackedUp) throws IOException {
    LOGGER.debug("Creating backup for distro: {}", muleDistribution);
    Path muleLocation = muleDistribution.getLocation();
    try {
      checkPossibleBackupOverwriting(muleLocation);
      purgeBackupDir(muleLocation);
      Path backupDestDir = muleLocation.resolve(upgradeConfigService.getBackupFolderName());
      fileSystemService.copyFiles(muleLocation, backupDestDir, relativePathOfFilesToBeBackedUp);
      generateDescriptor(muleDistribution);
    } catch (IOException e) {
      deleteBackupFolder(muleLocation);
      throw new IOException("Backup creation failed.", e);
    }
  }

  private void checkPossibleBackupOverwriting(Path muleLocation) throws IOException {
    List<String> messages = preconditionsValidatorService.checkBackupExistence(muleLocation);
    if (messages.isEmpty()) {
      printBackUpDescriptor(readBackupDescriptor(muleLocation));
      if (!preconditionsValidatorService.getConsentForOverwritingBackup(false)) {
        LOGGER.error("Consent is needed for overriding current backup.");
        throw new IllegalStateException("Consent is needed for overriding current backup.");
      }
    }
  }

  /**
   * This function reads the backup descriptor.
   *
   * @param muleDistributionHome location where is the backup descriptor.
   * @return BackupDescriptor or null.
   * @throws IOException
   */
  @Override
  public Optional<BackupDescriptor> readBackupDescriptor(Path muleDistributionHome) throws IOException {
    LOGGER.debug("Reading backup descriptor...");
    File yamlDescriptor = muleDistributionHome.resolve(getBackupDescriptorPath()).toFile();

    if (!yamlDescriptor.exists()) {
      LOGGER.debug("No backup descriptor found");
      return Optional.empty();
    }

    return Optional.of(yamlService.readValue(yamlDescriptor, BackupDescriptor.class));
  }

  /**
   * This function deletes current the backup.
   *
   * @param muleLocation location of the mule distribution backup.
   * @throws IOException
   */
  private void purgeBackupDir(Path muleLocation) throws IOException {
    File backupDestDir = muleLocation.resolve(upgradeConfigService.getBackupFolderName()).toFile();
    if (backupDestDir.exists()) {
      LOGGER.debug("Deleting backup folder at: {}", backupDestDir);
      FileUtils.forceDelete(backupDestDir);
    }
    createBackupDirectory(backupDestDir);
  }

  /**
   * This function creates the backup directory in client machine.
   *
   * @param backupDestDir directory to create in client machine.
   * @throws IOException
   */
  private void createBackupDirectory(File backupDestDir) throws IOException {
    backupDestDir.mkdirs();
    if (fileSystemService.isWindowsOs()) {
      Files.setAttribute(backupDestDir.toPath(), "dos:hidden", true);
    }
  }

  /**
   * Prints a BackUp Descriptor with a pretty format .
   *
   * @param backupDescriptor BackUp Descriptor selected.
   */
  public void printBackUpDescriptor(Optional<BackupDescriptor> backupDescriptor) {
    if (backupDescriptor.isPresent()) {
      String backupDescriptorPrettyFormatted = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
      LOGGER.info(backupDescriptorPrettyFormatted);
    }
  }

  /**
   * This function creates the backup descriptor.
   *
   * @param muleDistribution information about current mule distribution.
   * @throws IOException
   */
  private void generateDescriptor(MuleDistribution muleDistribution) throws IOException {
    String muleVersion = muleDistribution.getVersion().toString();
    Path muleLocation = muleDistribution.getLocation();
    BackupDescriptor backupDescriptor = builder()
        .backupGenerationTimestamp(Instant.now(getClock()))
        .upgradeToolVersion(appService.getVersion())
        .muleRuntimeDistributionVersion(muleVersion).build();

    File yamlDescriptor = muleLocation.resolve(getBackupDescriptorPath()).toFile();
    yamlService.writeValueToFile(yamlDescriptor, backupDescriptor);
  }

  /**
   * This function returns where is the backup descriptor to be saved.
   */
  private Path getBackupDescriptorPath() {
    return Paths.get(upgradeConfigService.getBackupFolderName(), upgradeConfigService.getBackupDescriptorFileName());
  }

  /**
   * This function deletes the backup directory.
   *
   * @param muleHome location of the backup directory.
   * @throws IOException
   */
  private void deleteBackupFolder(Path muleHome) throws IOException {
    File backupDir = muleHome.resolve(upgradeConfigService.getBackupFolderName()).toFile();
    if (backupDir.exists()) {
      FileUtils.forceDelete(backupDir);
    }
  }

  public Clock getClock() {
    if (clock == null) {
      clock = Clock.systemDefaultZone();
    }
    return clock;
  }

  public void setClock(Clock clock) {
    this.clock = clock;
  }
}
