/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.runtime.upgrade.tool.service;

import com.mulesoft.runtime.upgrade.tool.domain.AssemblyEntry;
import com.mulesoft.runtime.upgrade.tool.domain.enums.PathToBeExcluded;
import com.mulesoft.runtime.upgrade.tool.domain.enums.PathsToBeReplaced;
import com.mulesoft.runtime.upgrade.tool.service.api.DescriptorService;
import com.mulesoft.runtime.upgrade.tool.service.api.YamlService;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import org.apache.commons.io.FileUtils;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * This service manages file descriptor's functionalities.
 */
@Service
public class DefaultDescriptorService implements DescriptorService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultBackupService.class);

  @Autowired
  private YamlService yamlService;

  private static final String FILE_DESCRIPTOR_CURRENT_DISTRO = "file-descriptor-current-distro.yaml";
  private static final String FILE_DESCRIPTOR_NEW_DISTRO = "file-descriptor-new-distro.yaml";
  private static final String MULE_UPGRADE_TOOL =
      "tools/mule-runtime-upgrade-tool-1.0.0-SNAPSHOT.jar";


  private List<AssemblyEntry> entries = new ArrayList<>();

  /**
   * This method creates file descriptor of the current mule version and then compares with the newest file descriptor version.
   *
   * @param oldMulePath Path of the current mule distro.
   * @param newMulePath Path of the newest mule distro.
   * @throws Exception I/O Exception.
   */
  public void compareMuleDistros(Path oldMulePath, Path newMulePath) throws IOException {
    LOGGER.debug("Creating file descriptor of old distro...");
    createFileDescriptor(oldMulePath);
    File file = new File(FILE_DESCRIPTOR_CURRENT_DISTRO);
    file.deleteOnExit();
    LOGGER.debug("Comparing file descriptors...");
    compareMuleDistros(newMulePath);
  }

  private void compareMuleDistros(Path newMulePath) throws IOException {
    ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
    CollectionType listType = mapper.getTypeFactory().constructCollectionType(LinkedList.class, AssemblyEntry.class);

    List<AssemblyEntry> oldDistroFileDescriptor = mapper.readValue(new File(FILE_DESCRIPTOR_CURRENT_DISTRO), listType);
    List<AssemblyEntry> newDistroFileDescriptor =
        mapper.readValue(new File(newMulePath + "/" + FILE_DESCRIPTOR_NEW_DISTRO), listType);

    List<String> excludedPaths =
        PathToBeExcluded.getAllPaths().stream().map(x -> x.toFile().getName()).collect(Collectors.toList());

    oldDistroFileDescriptor.forEach(x -> {
      if (!newDistroFileDescriptor.contains(x) && !x.getName().equals(MULE_UPGRADE_TOOL)
          && !FilenameUtils.getExtension(x.getName()).equals("lic") && !excludedPaths.contains(x.getName())) {
        LOGGER.warn(x.getName() + " will be deleted.");
      }
    });
  }

  private void createFileDescriptor(Path mulePath) throws IOException {

    List<Path> allPaths = PathsToBeReplaced.getAllPaths();

    for (Path path : allPaths) {
      try {
        recursiveFileTreeWrapper(mulePath.resolve(path));
      } catch (IOException e) {
        throw new IOException(path + " was not found.", e);
      }
    }


    try {
      yamlService.writeValueToFile(new File(FILE_DESCRIPTOR_CURRENT_DISTRO), entries);
    } catch (IOException e) {
      throw new IOException("Error generating the distribution assembly content descriptor", e);
    }
  }

  private void extractEntryInfo(File file, String parentDirectory) throws IOException {
    try {
      byte[] contents = FileUtils.readFileToByteArray(file);
      String sha256 = DigestUtils.sha256Hex(contents);
      entries.add(new AssemblyEntry(parentDirectory, contents.length, sha256));
    } catch (IOException e) {
      throw new IOException("Error generating the Mule distribution assembly descriptor", e);
    }
  }

  private void recursiveFileTreeWrapper(Path directory) throws IOException {
    recursiveFileTree(directory, directory.toFile().getName());
  }

  private void recursiveFileTree(Path path, String parent) throws IOException {

    if (path == null)
      return;

    File file = path.toFile();

    if (file.isFile()) {
      extractEntryInfo(file, parent);
      return;
    }
    if (file.isDirectory()) {
      for (File elem : file.listFiles()) {
        recursiveFileTree(elem.toPath(), parent + "/" + elem.getName());
      }
    }
  }
}
