/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.service.utils.PathDefinitionUtils.getAllDefinedPaths;

import com.mulesoft.runtime.upgrade.tool.domain.RollbackOptions;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.PreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.RollbackService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;
import com.mulesoft.runtime.upgrade.tool.service.api.UserInteractionService;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * Service to rollback a Mule Runtime distribution upgrade from a backup.
 */
@Service
public class DefaultRollbackService implements RollbackService {

  public static final String USER_REJECTED_TO_PROCEED_WITH_THE_ROLLBACK_PROCESS_MESSAGE =
      "User has rejected to proceed with the rollback process";

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultRollbackService.class);

  @Autowired
  private MuleDistroService muleDistroService;

  @Autowired
  private PreconditionsValidatorService preconditionsValidatorService;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private UserInteractionService userInteractionService;

  /**
   * Rolls back a Mule Runtime distribution upgrade from a backup if any exists. Copies back all the files stored in the backup to
   * their original location, leaving the Mule distribution in the same state as when the backup was created. The files are taken
   * out of the backup folder according to the upgrade configuration, in a sort of cherry-picked fashion rather than a bulk dump
   * copy of everything under the backup location.
   *
   * @param muleDistroLocation path to the Mule distribution to be roll-back.
   * @param rollbackOptions    configuration options for the rollback process.
   *
   * @throws IOException           if an I/O error occurs.
   * @throws IllegalStateException if there's no backup for the rollback of the given Mule distro.
   */
  @Override
  public void rollback(Path muleDistroLocation, RollbackOptions rollbackOptions) throws IOException {
    LOGGER.info("Preparing rollback process...");

    muleDistroService.looksLikeADistro(muleDistroLocation);

    List<String> possibleErrors = preconditionsValidatorService.checkRollbackPreconditions(muleDistroLocation);

    if (!possibleErrors.isEmpty())
      throw new IllegalStateException("Not all preconditions were met for performing the rollback. Review the log for details.");

    muleDistroService.checkClusterModeIsActive(muleDistroLocation);
    requestUsersConfirmation(rollbackOptions.isForce());

    try {
      LOGGER.info("Rolling back the Mule Runtime distribution.");
      copyBackFiles(muleDistroLocation);
      LOGGER.info("Rollback process finished successfully.");
    } catch (IOException e) {
      throw new IOException("Failed to copy back files from the backup. Consider re-trying the operation after having solved " +
          "the underlying failure condition, if any", e);
    }
  }

  private void requestUsersConfirmation(boolean force) throws IOException {
    if (!force && !userInteractionService.confirmAction("rollback upgrade", "Do you want to proceed with the rollback?", false)) {
      throw new IllegalStateException(USER_REJECTED_TO_PROCEED_WITH_THE_ROLLBACK_PROCESS_MESSAGE);
    }
  }

  protected void copyBackFiles(Path distroLocation) throws IOException {
    Path backupFolder = distroLocation.resolve(upgradeConfigService.getBackupFolderName());
    LOGGER.debug("Copying backed up files from [{}] to [{}]...", backupFolder, distroLocation);

    List<Path> expectedFilesInTheBackup = getAllDefinedPaths();
    fileSystemService.copyFiles(backupFolder, distroLocation, expectedFilesInTheBackup);
  }
}
