/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.domain.enums.RollbackPrecondition.BACKUP;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.RollbackPrecondition.BACKUP_INTEGRITY;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.RollbackPrecondition.MULE_NOT_RUNNING;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.RollbackPrecondition.WRITING_PERMISSIONS;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.CONSENT_TO_OVERWRITE_BACKUP;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.CONSENT_TO_REMOVE_APPLIED_PATCHES;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.FILE_SYSTEM_SPACE;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.NEW_MULE_IS_NEWER;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.NEW_MULE_READING_PERMISSIONS;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.OLD_MULE_NOT_RUNNING;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.OLD_MULE_WRITING_PERMISSIONS;
import static com.mulesoft.runtime.upgrade.tool.service.utils.PathDefinitionUtils.getAllDefinedPaths;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatAppliedPatches;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatBackupDescriptor;
import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.domain.enums.RollbackPrecondition;
import com.mulesoft.runtime.upgrade.tool.domain.SemVer;
import com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.DistroPatchService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleStatusService;
import com.mulesoft.runtime.upgrade.tool.service.api.PreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.UserInteractionService;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A preconditions validator for the tool's main processes.
 */
@Service
public class DefaultPreconditionsValidatorService implements PreconditionsValidatorService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultPreconditionsValidatorService.class);

  @Autowired
  private BackupService backupService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private MuleStatusService statusService;

  @Autowired
  private UserInteractionService userInteractionService;

  @Autowired
  private DistroPatchService patchService;

  /**
   * Checks that all preconditions for performing the upgrade are met.
   *
   * <p>
   * Preconditions are:
   * <ul>
   * <li>Distribution marked as new is, in fact, a newer version of the distribution being upgraded.</li>
   * <li>Process has writing permissions under the old Mule distribution location.</li>
   * <li>Process has reading permissions under the new Mule distribution location.</li>
   * <li>Enough space is available in the file system for the upgrade.</li>
   * <li>Old Mule Distribution is stopped.</li>
   * <li>Consent for overriding backup has been provided, if any.</li>
   * <li>Consent for removing applied patches has been provided, if any.</li>
   * </ul>
   *
   * @param oldMule {@link MuleDistribution} representing the old Mule Runtime distribution being upgraded
   * @param newMule {@link MuleDistribution} representing the new Mule Runtime distribution used for the upgrade
   * @param force   whether explicit consent has been provided for the override related preconditions like the ones for the backup
   *                and applied patches
   *
   * @throws IllegalStateException if validations are not met
   * @throws IOException           if an I/O error occurs
   */
  @Override
  public void checkUpgradeReadiness(MuleDistribution oldMule, MuleDistribution newMule, boolean force) throws IOException {
    LOGGER.debug("Validating if all preconditions to proceed with the upgrade are met...");
    Map<UpgradePrecondition, Boolean> preconditionsMet = new EnumMap<>(UpgradePrecondition.class);

    preconditionsMet.put(NEW_MULE_IS_NEWER, checkNewMuleIsNewer(oldMule.getVersion(), newMule.getVersion()));
    preconditionsMet.put(OLD_MULE_WRITING_PERMISSIONS, checkWritingPermissions(oldMule.getLocation()));
    preconditionsMet.put(NEW_MULE_READING_PERMISSIONS, checkReadingPermissions(newMule.getLocation()));
    preconditionsMet.put(FILE_SYSTEM_SPACE, checkEnoughUsableSpace(oldMule.getLocation(), newMule.getLocation()));
    preconditionsMet.put(OLD_MULE_NOT_RUNNING, isMuleStopped(oldMule.getLocation()));
    preconditionsMet.put(CONSENT_TO_OVERWRITE_BACKUP, getConsentForOverwritingBackup(oldMule.getLocation(), force));
    preconditionsMet.put(CONSENT_TO_REMOVE_APPLIED_PATCHES, getConsentForRemovingAppliedPatches(oldMule.getLocation(), force));

    if (preconditionsMet.values().contains(false)) {
      throw new IllegalStateException("Not all preconditions were met for performing the upgrade. Review the log for details.");
    }
  }

  private boolean checkNewMuleIsNewer(SemVer oldDistro, SemVer newDistro) {
    if (oldDistro.compareTo(newDistro) >= 0) {
      LOGGER.error("Version of new Mule distribution {} should be newer than old Mule distribution: {}", newDistro, oldDistro);
      return false;
    }
    return true;
  }

  /**
   * Checks that all preconditions for performing the rollback are met.
   *
   * <p>
   * Preconditions are:
   * <ul>
   * <li>The existence of a backup</li>
   * <li>Process has writing permissions under the old Mule distribution location.</li>
   * <li>Mule Distribution is stopped.</li>
   * </ul>
   *
   * @param muleDistroLocation location of the Mule Runtime distribution being validated for rollback process.
   *
   * @throws IllegalStateException if validations are not met.
   * @throws IOException           if an I/O error occurs.
   */
  @Override
  public void checkRollbackReadiness(Path muleDistroLocation) throws IOException {
    LOGGER.debug("Validating if all preconditions to proceed with the rollback are met...");
    Map<RollbackPrecondition, Boolean> preconditionsMet = new EnumMap<>(RollbackPrecondition.class);

    preconditionsMet.put(BACKUP, checkBackupExistence(muleDistroLocation));
    preconditionsMet.put(BACKUP_INTEGRITY, checkBackupIntegrity(muleDistroLocation));
    preconditionsMet.put(WRITING_PERMISSIONS, checkWritingPermissions(muleDistroLocation));
    preconditionsMet.put(MULE_NOT_RUNNING, isMuleStopped(muleDistroLocation));

    if (preconditionsMet.values().contains(false)) {
      throw new IllegalStateException("Not all preconditions were met for performing the rollback. Review the log for details.");
    }
  }

  private boolean getConsentForOverwritingBackup(Path distroLocation, boolean force) throws IOException {
    Optional<BackupDescriptor> backupDescriptor = backupService.readBackupDescriptor(distroLocation);

    if (!backupDescriptor.isPresent()) {
      LOGGER.debug("No existent backup found for Mule Runtime distribution: [{}]", distroLocation);
      return true;
    } else {
      LOGGER.info("Backup found for Mule Runtime distribution: [{}]", distroLocation);
      String backupDescriptorPrettyFormatted = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
      LOGGER.info(backupDescriptorPrettyFormatted);
      if (force) {
        LOGGER.debug("Consent to overwrite backup is set through force parameter.");
        return true;
      } else {
        return userInteractionService.confirmAction("overwrite backup", "Could backup be overwritten?", false);
      }
    }
  }

  private boolean getConsentForRemovingAppliedPatches(Path distroLocation, boolean force) throws IOException {
    List<File> appliedPatches = patchService.getAppliedPatches(distroLocation);

    if (appliedPatches.isEmpty()) {
      LOGGER.debug("No applied patches found for Mule Runtime distribution: [{}]", distroLocation);
      return true;
    } else {
      LOGGER.info("Applied patches found for Mule Runtime distribution: [{}]", distroLocation);
      String appliedPatchesListPrettyFormatted = format("%n%s", formatAppliedPatches(appliedPatches));
      LOGGER.info(appliedPatchesListPrettyFormatted);
      if (force) {
        LOGGER.debug("Consent to remove applied patches is set through force parameter.");
        return true;
      } else {
        return userInteractionService.confirmAction("remove patches", "Could patched be removed?", false);
      }
    }
  }

  public boolean checkBackupExistence(Path muleDistroLocation) throws IOException {
    Optional<BackupDescriptor> backupDescriptor = backupService.readBackupDescriptor(muleDistroLocation);
    if (!backupDescriptor.isPresent()) {
      LOGGER.error("Backup does not exist for Mule Runtime distribution: [{}]", muleDistroLocation);
    } else {
      LOGGER.debug("Backup found for Mule Runtime distribution: [{}]", muleDistroLocation);
      String backUpDescriptorPrettyFormatted = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
      LOGGER.debug(backUpDescriptorPrettyFormatted);
    }
    return backupDescriptor.isPresent();
  }

  public List<String> getBackupIntegrityErrorMessages(Path muleDistroLocation) {
    List<String> errorMessages = new ArrayList<>();
    String backupErrorMessage =
        String.format("Backup integrity check wasn't successful for Mule Runtime distribution: [%s]", muleDistroLocation);
    errorMessages.add(backupErrorMessage);
    return errorMessages;
  }

  public boolean checkBackupIntegrity(Path muleDistroLocation) throws IOException {
    if (backupService.checkIntegrity(muleDistroLocation)) {
      LOGGER.debug("Backup integrity check was successful for Mule Runtime distribution: [{}]", muleDistroLocation);
      return true;
    } else {
      getBackupIntegrityErrorMessages(muleDistroLocation).stream().forEachOrdered(
                                                                                  LOGGER::error);
      return false;
    }
  }

  private boolean checkWritingPermissions(Path muleDistroLocation) {
    try {
      fileSystemService.checkWritingAccess(muleDistroLocation, getAllDefinedPaths());
      LOGGER.debug("Writing permissions found for all expected paths within the Mule distribution [{}]", muleDistroLocation);
      return true;
    } catch (IOException e) {
      LOGGER.error(format("Error validating required writing permissions under Mule distribution [%s]", muleDistroLocation), e);
      return false;
    }
  }

  private boolean checkEnoughUsableSpace(Path oldMule, Path newMule) {
    try {
      fileSystemService.checkEnoughUsableSpace(oldMule, newMule, getAllDefinedPaths());
      LOGGER.debug("There's enough disk space for performing the upgrade.");
      return true;
    } catch (IOException e) {
      LOGGER.error("There's not enough disk space for performing the upgrade.");
      return false;
    }
  }

  private boolean checkReadingPermissions(Path muleDistroLocation) {
    try {
      fileSystemService.checkReadingAccess(muleDistroLocation, getAllDefinedPaths());
      LOGGER.debug("Reading permissions found for all expected paths within the Mule distribution [{}]", muleDistroLocation);
      return true;
    } catch (IOException e) {
      LOGGER.error(format("Error validating required reading permissions under Mule distribution [%s]", muleDistroLocation), e);
      return false;
    }
  }

  private boolean isMuleStopped(Path muleDistroLocation) throws IOException {
    try {
      statusService.checkStopped(muleDistroLocation);
    } catch (IllegalStateException e) {
      LOGGER.error("Mule Runtime should be stopped.");
      return false;
    }
    LOGGER.debug("Mule Distribution located at [{}] is stopped.", muleDistroLocation);
    return true;
  }
}
