/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.FILE_SYSTEM_SPACE;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.NEW_MULE_IS_NEWER;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.NEW_MULE_READING_PERMISSIONS;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.OLD_MULE_NOT_RUNNING;
import static com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition.OLD_MULE_WRITING_PERMISSIONS;
import static com.mulesoft.runtime.upgrade.tool.service.utils.PathDefinitionUtils.getAllDefinedPaths;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatAppliedPatches;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatBackupDescriptor;
import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.domain.SemVer;
import com.mulesoft.runtime.upgrade.tool.domain.enums.UpgradePrecondition;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.DistroPatchService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleStatusService;
import com.mulesoft.runtime.upgrade.tool.service.api.StatusPreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.UserInteractionService;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.EnumMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A preconditions validator for the tool's main processes.
 */
@Service
public class DefaultStatusPreconditionsValidatorService implements StatusPreconditionsValidatorService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultStatusPreconditionsValidatorService.class);

  @Autowired
  private BackupService backupService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private MuleStatusService statusService;

  @Autowired
  private UserInteractionService userInteractionService;

  @Autowired
  private DistroPatchService patchService;

  @Autowired
  private DefaultPreconditionsValidatorService preconditionsValidatorService;

  /**
   * Checks that all preconditions for performing the upgrade are met.
   *
   * <p>
   * Preconditions are:
   * <ul>
   * <li>Distribution marked as new is, in fact, a newer version of the distribution being upgraded.</li>
   * <li>Process has writing permissions under the old Mule distribution location.</li>
   * <li>Process has reading permissions under the new Mule distribution location.</li>
   * <li>Enough space is available in the file system for the upgrade.</li>
   * <li>Old Mule Distribution is stopped.</li>
   * <li>Consent for overriding backup has been provided, if any.</li>
   * <li>Consent for removing applied patches has been provided, if any.</li>
   * </ul>
   *
   * @param oldMule {@link MuleDistribution} representing the old Mule Runtime distribution being upgraded
   * @param newMule {@link MuleDistribution} representing the new Mule Runtime distribution used for the upgrade
   * @param force   whether explicit consent has been provided for the override related preconditions like the ones for the backup
   *                and applied patches
   * @throws IllegalStateException if validations are not met
   * @throws IOException           if an I/O error occurs
   */
  @Override
  public void checkUpgradePreconditions(MuleDistribution oldMule, MuleDistribution newMule, boolean force)
      throws IOException {

    LOGGER.debug("Validating if all preconditions to proceed with the upgrade are met...");
    Map<UpgradePrecondition, Boolean> preconditionsMet = new EnumMap<>(UpgradePrecondition.class);

    preconditionsMet.put(NEW_MULE_IS_NEWER, checkNewMuleIsNewer(oldMule.getVersion(), newMule.getVersion()).isEmpty());
    preconditionsMet.put(OLD_MULE_WRITING_PERMISSIONS, checkWritingPermissions(oldMule.getLocation()).isEmpty());
    preconditionsMet.put(NEW_MULE_READING_PERMISSIONS, checkReadingPermissions(newMule.getLocation()).isEmpty());
    preconditionsMet.put(FILE_SYSTEM_SPACE, checkEnoughUsableSpace(oldMule.getLocation(), newMule.getLocation()).isEmpty());
    preconditionsMet.put(OLD_MULE_NOT_RUNNING, isMuleStopped(oldMule.getLocation()).isEmpty());

    if (preconditionsMet.values().contains(false)) {
      throw new IllegalStateException("Not all preconditions were met for performing the upgrade. Review the log for details.");
    }
  }

  private boolean getConsentForRemovingAppliedPatches(Path distroLocation, boolean force) throws IOException {
    List<File> appliedPatches = patchService.getAppliedPatches(distroLocation);

    if (appliedPatches.isEmpty()) {
      LOGGER.debug("No applied patches found for Mule Runtime distribution: [{}]", distroLocation);
      return true;
    } else {
      LOGGER.info("Applied patches found for Mule Runtime distribution: [{}]", distroLocation);
      String appliedPatchesListPrettyFormatted = format("%n%s", formatAppliedPatches(appliedPatches));
      LOGGER.info(appliedPatchesListPrettyFormatted);
      if (force) {
        LOGGER.debug("Consent to remove applied patches is set through force parameter.");
        return true;
      } else {
        return userInteractionService.confirmAction("remove patches", "Could patched be removed?", false);
      }
    }
  }

  /**
   * Checks upgrade pre-conditions for status command.
   *
   * @param oldMule {@link MuleDistribution} representing the old Mule Runtime distribution being upgraded.
   * @throws IOException if an I/O error occurs
   */
  @Override
  public List<String> checkUpgradeStatusOffline(MuleDistribution oldMule)
      throws IOException {

    LOGGER.debug("Checking pre-conditions status for upgrade process...");

    List<String> statusErrorMessages = new LinkedList<>();
    statusErrorMessages.add("Checking upgrade process pre-conditions...");

    statusErrorMessages.addAll(checkWritingPermissions(oldMule.getLocation()));
    statusErrorMessages.addAll(isMuleStopped(oldMule.getLocation()));

    return statusErrorMessages;
  }


  /**
   * Checks rollback pre-conditions for status command.
   *
   * @param muleDistroLocation {@link MuleDistribution} representing the current Mule Runtime distribution location.
   * @throws IOException if an I/O error occurs
   */
  @Override
  public List<String> checkRollbackPreconditions(Path muleDistroLocation) throws IOException {
    LOGGER.debug("Validating if all preconditions to proceed with the rollback are met...");

    List<String> statusErrorMessages = new LinkedList<>();

    statusErrorMessages.addAll(checkBackupExistence(muleDistroLocation));
    if (!preconditionsValidatorService.checkBackupIntegrity(muleDistroLocation)) {
      statusErrorMessages.addAll(preconditionsValidatorService.getBackupIntegrityErrorMessages(muleDistroLocation));
    }
    statusErrorMessages.addAll(checkWritingPermissions(muleDistroLocation));
    statusErrorMessages.addAll(isMuleStopped(muleDistroLocation));

    return statusErrorMessages;
  }

  private List<String> checkNewMuleIsNewer(SemVer oldDistro, SemVer newDistro) {
    List<String> messages = new LinkedList<>();
    if (oldDistro.compareTo(newDistro) >= 0) {
      String message = String.format("Version of new Mule distribution {} should be newer than old Mule distribution: {}",
                                     newDistro, oldDistro);
      messages.add(message);
      LOGGER.error(message);
    }
    return messages;
  }

  /**
   * Asks consent for overwriting the current BackUp.
   *
   * @param force If force is true, will not ask to the user.
   * @return User response.
   * @throws IOException I/O Exception.
   */
  public boolean getConsentForOverwritingBackup(boolean force) throws IOException {
    LOGGER.info("Asking for consent for overriding backup...");

    if (force) {
      LOGGER.debug("Consent to overwrite backup is set through force parameter.");
      return true;
    } else {
      return userInteractionService.confirmAction("overwrite backup", "Could backup be overwritten?", false);
    }
  }

  /**
   * Checks existence of any BackUp.
   *
   * @param muleDistroLocation Mule path to looking for the BackUp.
   * @return A list of Strings, if the list is empty, the BackUp exists, otherwise not exists.
   * @throws IOException I/O Exception.
   */
  public List<String> checkBackupExistence(Path muleDistroLocation) throws IOException {
    Optional<BackupDescriptor> backupDescriptor = backupService.readBackupDescriptor(muleDistroLocation);
    List<String> messages = new LinkedList<>();

    if (!backupDescriptor.isPresent()) {
      String message = String.format("Backup does not exist for Mule Runtime distribution: [%s].", muleDistroLocation);
      messages.add(message);
    } else {
      LOGGER.debug("Backup found for Mule Runtime distribution: [{}]", muleDistroLocation);
      String backUpDescriptorPrettyFormatted = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
      LOGGER.debug(backUpDescriptorPrettyFormatted);
    }
    return messages;
  }

  private List<String> checkWritingPermissions(Path muleDistroLocation) {
    List<String> messages = new LinkedList<>();
    try {
      fileSystemService.checkWritingAccess(muleDistroLocation, getAllDefinedPaths());
      LOGGER.debug("Writing permissions found for all expected paths within the Mule distribution [{}]", muleDistroLocation);
    } catch (IOException e) {
      String message =
          String.format("Error validating required writing permissions under Mule distribution [%s].", muleDistroLocation);
      messages.add(message);
      LOGGER.error(message, e);
    }
    return messages;
  }

  private List<String> checkEnoughUsableSpace(Path oldMule, Path newMule) {
    List<String> messages = new LinkedList<>();
    try {
      fileSystemService.checkEnoughUsableSpace(oldMule, newMule, getAllDefinedPaths());
      LOGGER.debug("There's enough disk space for performing the upgrade.");
    } catch (IOException e) {
      String message = "There's not enough disk space for performing the upgrade.";
      messages.add(message);
      LOGGER.error(message);
    }
    return messages;
  }

  private List<String> checkReadingPermissions(Path muleDistroLocation) {
    List<String> messages = new LinkedList<>();
    try {
      fileSystemService.checkReadingAccess(muleDistroLocation, getAllDefinedPaths());
      LOGGER.debug("Reading permissions found for all expected paths within the Mule distribution [{}]", muleDistroLocation);
    } catch (IOException e) {
      String message =
          String.format("Error validating required reading permissions under Mule distribution [%s].", muleDistroLocation);
      messages.add(message);
      LOGGER.error(message, e);
    }
    return messages;
  }

  private List<String> isMuleStopped(Path muleDistroLocation) throws IOException {
    List<String> messages = new LinkedList<>();
    try {
      statusService.checkStopped(muleDistroLocation);
    } catch (IllegalStateException e) {
      String message = "Mule Runtime should be stopped.";
      messages.add(message);
      LOGGER.error(message);
    }
    LOGGER.debug("Mule Distribution located at [{}] is stopped.", muleDistroLocation);
    return messages;
  }
}
