/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.fasterxml.jackson.databind.PropertyNamingStrategies.KEBAB_CASE;
import static com.fasterxml.jackson.databind.SerializationFeature.WRITE_DATES_AS_TIMESTAMPS;
import static com.fasterxml.jackson.dataformat.yaml.YAMLGenerator.Feature.WRITE_DOC_START_MARKER;

import com.mulesoft.runtime.upgrade.tool.service.api.YamlService;

import java.io.File;
import java.io.IOException;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

/**
 * Service for marshalling and unmarshalling of values in YAML format.
 */
@Service
public class DefaultYamlService implements YamlService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultYamlService.class);

  /**
   * Serializes any Java value as YAML in the given file.
   *
   * @param resultFile output file.
   * @param object     object value.
   *
   * @throws IOException If serialization operation fails.
   */
  @Override
  public void writeValueToFile(File resultFile, Object object) throws IOException {
    LOGGER.debug("Writing object to YAML file at {}", resultFile);
    buildYamlObjectMapper().writeValue(resultFile, object);
  }

  /**
   * Deserializes a YAML value to a Java one from a given file.
   *
   * @param src       the source file.
   * @param valueType the class of the output value type.
   * @return
   * @throws IOException if a low-level I/O problem occurs while deserializing the YAML value.
   */
  @Override
  public <T> T readValue(File src, Class<T> valueType) throws IOException {
    LOGGER.debug("Reading value from YAML file at {}", src);
    return buildYamlObjectMapper().readValue(src, valueType);
  }

  private ObjectMapper buildYamlObjectMapper() {
    return new ObjectMapper(new YAMLFactory().disable(WRITE_DOC_START_MARKER))
        .setPropertyNamingStrategy(KEBAB_CASE)
        .disable(WRITE_DATES_AS_TIMESTAMPS)
        .registerModule(new JavaTimeModule());
  }
}
