/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service.api;

import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

/**
 * A preconditions validator for the tool main processes.
 */
public interface StatusPreconditionsValidatorService {

  /**
   * Checks that all preconditions for performing the upgrade are met.
   *
   * @param oldMule {@link MuleDistribution} representing the old Mule Runtime distribution being upgraded.
   * @param newMule {@link MuleDistribution} representing the new Mule Runtime distribution used for the upgrade.
   * @param force   whether explicit consent has been provided for the override related preconditions like the ones for the backup
   *                and applied patches.
   *
   * @throws IllegalStateException if validations are not met.
   * @throws IOException           if an I/O error occurs.
   */
  void checkUpgradePreconditions(MuleDistribution oldMule, MuleDistribution newMule, boolean force)
      throws IOException;

  /**
   * Checks that all preconditions for performing the rollback are met.
   *
   * @param muleDistroLocation location of the Mule Runtime distribution being validated for rollback process.
   *
   * @throws IllegalStateException if validations are not met.
   * @throws IOException           if an I/O error occurs.
   */
  List<String> checkRollbackPreconditions(Path muleDistroLocation) throws IOException;

  /**
   * Checks upgrade pre-conditions for status command.
   *
   * @param oldMule {@link MuleDistribution} representing the old Mule Runtime distribution being upgraded.
   *
   * @throws IOException if an I/O error occurs
   */
  List<String> checkUpgradeStatusOffline(MuleDistribution oldMule) throws IOException;

  /**
   * Asks consent for overwriting the current BackUp.
   *
   * @param force If force is true, will not ask to the user.
   *
   * @return User response.
   *
   * @throws IOException I/O Exception.
   */
  boolean getConsentForOverwritingBackup(boolean force) throws IOException;

  /**
   * Checks existence of any BackUp.
   *
   * @param muleDistroLocation Mule path to looking for the BackUp.
   *
   * @return A list of Strings, if the list is empty, the BackUp exists, otherwise not exists.
   *
   * @throws IOException I/O Exception.
   */
  List<String> checkBackupExistence(Path muleDistroLocation) throws IOException;
}
