/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.utils;

import static java.lang.String.format;
import static java.util.Comparator.comparingInt;
import static java.util.stream.Collectors.joining;
import static java.util.stream.IntStream.rangeClosed;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Class for creating pretty String representation of domain objects like the backup descriptor or applied patches list that could
 * be used for formatted printing in a writer.
 */
public class PrettyPrintingFormatter {

  private PrettyPrintingFormatter() {
    // sonarqube RSPEC-1118
    throw new IllegalStateException("Utility class");
  }

  public static String formatBackupDescriptor(BackupDescriptor backupDescriptor) {
    List<String> formattedLines = new ArrayList<>();
    formattedLines.add("*** Backup metadata:");
    formattedLines.add(format("- Generation timestamp: %s", backupDescriptor.getBackupGenerationTimestamp()));
    formattedLines.add(format("- Upgrade tool version: %s", backupDescriptor.getUpgradeToolVersion()));
    formattedLines.add(format("- Mule Runtime distribution version: %s", backupDescriptor.getMuleRuntimeDistributionVersion()));
    return prettify(formattedLines);
  }

  public static String formatStatusCommandMessage(List<String> messages) {
    List<String> formattedLines = new ArrayList<>();
    formattedLines.add("*** STATUS:");
    formattedLines.addAll(messages);
    return prettify(formattedLines);
  }

  public static String formatAppliedPatches(List<File> appliedPatches) {
    List<String> formattedLines = new ArrayList<>();
    formattedLines.add("*** Applied patches list:");
    appliedPatches.forEach(file -> formattedLines.add(format("- %s", file.getName())));
    return prettify(formattedLines);
  }

  private static String prettify(List<String> formattedLines) {
    int maxLineLength = formattedLines.stream().max(comparingInt(String::length)).orElseGet(String::new).length();
    String lineSeparator = format("*** %s%n", rangeClosed(0, maxLineLength - 4).mapToObj(x -> "*").collect(joining()));
    StringBuilder builder = new StringBuilder();

    // the header
    builder.append(lineSeparator);
    builder.append(format("%s%n", formattedLines.get(0)));
    builder.append(lineSeparator);
    // the body
    formattedLines.stream().skip(1).forEach(str -> builder.append(format("%s%n", str)));
    // the end
    builder.append(lineSeparator);

    return builder.toString();
  }

}
