/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor.builder;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatBackupDescriptor;
import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.domain.enums.PathToBeExcluded;
import com.mulesoft.runtime.upgrade.tool.service.api.AppService;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.DescriptorService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.StatusPreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;
import com.mulesoft.runtime.upgrade.tool.service.api.YamlService;
import com.mulesoft.runtime.upgrade.tool.service.utils.PathDefinitionUtils;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A service for backup related operations.
 */
@Service
public class DefaultBackupService implements BackupService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultBackupService.class);

  @Autowired
  private AppService appService;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  @Autowired
  private YamlService yamlService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private DescriptorService descriptorService;

  @Autowired
  private StatusPreconditionsValidatorService statusPreconditionsValidatorService;

  // For managing the time in tests
  private Clock clock;

  /**
   * This function creates the backup descriptor.
   *
   * @param muleDistribution a {@link MuleDistribution} representing the Mule distro where the backup will be taken from.
   * @param isDryRunMode     if isDryRunMode is true, the tool only log in debug level the files and directories that would copy
   *                         but not copy them properly.
   * @throws IOException
   */
  public void create(MuleDistribution muleDistribution, boolean isDryRunMode) throws IOException {
    create(muleDistribution, PathDefinitionUtils.getAllDefinedPaths(), PathToBeExcluded.getAllPaths(), isDryRunMode);
  }

  private void create(MuleDistribution muleDistribution, List<Path> relativePathOfFilesToBeBackedUp,
                      List<Path> relativePathOfFilesToBeExcluded, boolean isDryRunMode)
      throws IOException {
    LOGGER.debug("Creating backup for distro: {}", muleDistribution);
    Path muleLocation = muleDistribution.getLocation();
    try {
      checkPossibleBackupOverwriting(muleLocation);
      purgeBackupDir(muleLocation, isDryRunMode);
      Path backupDestDir = getBackupDestDir(muleLocation);
      fileSystemService.copyFiles(muleLocation, backupDestDir, relativePathOfFilesToBeBackedUp, relativePathOfFilesToBeExcluded,
                                  isDryRunMode);
      generateDescriptor(muleDistribution, backupDestDir, isDryRunMode);
    } catch (IOException e) {
      deleteBackupFolder(muleLocation, isDryRunMode);
      throw new IOException("Backup creation failed.", e);
    }
  }

  private void checkPossibleBackupOverwriting(Path muleLocation) throws IOException {
    List<String> errorMessages = statusPreconditionsValidatorService.checkBackupExistence(muleLocation);
    if (errorMessages.isEmpty()) {
      printBackUpDescriptor(readBackupDescriptor(muleLocation));
      if (!statusPreconditionsValidatorService.getConsentForOverwritingBackup(false)) {
        LOGGER.error("Consent is needed for overriding current backup.");
        throw new IllegalStateException("Consent is needed for overriding current backup.");
      }
    }
  }

  /**
   * This function reads the backup descriptor.
   *
   * @param muleDistributionHome location where is the backup descriptor.
   * @return BackupDescriptor or null.
   * @throws IOException
   */
  @Override
  public Optional<BackupDescriptor> readBackupDescriptor(Path muleDistributionHome) throws IOException {
    LOGGER.debug("Reading backup descriptor...");
    File yamlDescriptor = getBackupDescriptorPathForDistro(muleDistributionHome).toFile();

    if (!yamlDescriptor.exists()) {
      LOGGER.debug("No backup descriptor found");
      return Optional.empty();
    }

    return Optional.of(yamlService.readValue(yamlDescriptor, BackupDescriptor.class));
  }

  @Override
  public boolean checkIntegrity(Path muleDistributionHome) throws IOException {
    LOGGER.debug("Checking backup integrity...");
    Optional<BackupDescriptor> backupDescriptor = readBackupDescriptor(muleDistributionHome);

    if (backupDescriptor.isPresent()) {
      return descriptorService.checkIntegrity(backupDescriptor.get().getBackupEntries(), getBackupDestDir(muleDistributionHome));
    } else {
      LOGGER.debug("As the backup descriptor was not found, the backup integrity check was not done.");
      return false;
    }
  }

  Path getBackupDestDir(Path muleLocation) {
    return muleLocation.resolve(upgradeConfigService.getBackupFolderName());
  }

  private Path getBackupDescriptorPathForDistro(Path muleDistroHome) {
    return muleDistroHome.resolve(getBackupDescriptorPath());
  }

  /**
   * This function deletes current the backup.
   *
   * @param muleLocation location of the mule distribution backup.
   * @param isDryRunMode if isDryRunMode is true, the tool will not modify the backup.
   * @throws IOException
   */
  private void purgeBackupDir(Path muleLocation, boolean isDryRunMode) throws IOException {
    File backupDestDir = getBackupDestDir(muleLocation).toFile();
    if (backupDestDir.exists()) {
      LOGGER.debug("Deleting backup folder at: {}", backupDestDir);
      if (!isDryRunMode) {
        FileUtils.forceDelete(backupDestDir);
      }
    }
    if (!isDryRunMode) {
      createBackupDirectory(backupDestDir);
    }
  }

  /**
   * This function creates the backup directory in client machine.
   *
   * @param backupDestDir directory to create in client machine.
   * @throws IOException
   */
  private void createBackupDirectory(File backupDestDir) throws IOException {
    backupDestDir.mkdirs();
    if (fileSystemService.isWindowsOs()) {
      Files.setAttribute(backupDestDir.toPath(), "dos:hidden", true);
    }
  }

  /**
   * Prints a BackUp Descriptor with a pretty format .
   *
   * @param backupDescriptor BackUp Descriptor selected.
   */
  public void printBackUpDescriptor(Optional<BackupDescriptor> backupDescriptor) {
    if (backupDescriptor.isPresent()) {
      String backupDescriptorPrettyFormatted = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
      LOGGER.info(backupDescriptorPrettyFormatted);
    }
  }

  /**
   * This function creates the backup descriptor.
   *
   * @param muleDistribution information about current mule distribution.
   * @param isDryRunMode     if isDryRunMode is true, the tool will not generate the descriptor.
   * @throws IOException
   */
  private void generateDescriptor(MuleDistribution muleDistribution, Path backupPath, boolean isDryRunMode) throws IOException {
    String muleVersion = muleDistribution.getVersion().toString();
    Path muleLocation = muleDistribution.getLocation();
    BackupDescriptor backupDescriptor = builder()
        .backupGenerationTimestamp(Instant.now(getClock()))
        .upgradeToolVersion(appService.getVersion())
        .muleRuntimeDistributionVersion(muleVersion)
        .backupEntries(descriptorService.getEntriesInPath(backupPath))
        .build();

    File yamlDescriptor = muleLocation.resolve(getBackupDescriptorPath()).toFile();

    if (!isDryRunMode) {
      yamlService.writeValueToFile(yamlDescriptor, backupDescriptor);
    }
  }

  /**
   * This function returns where is the backup descriptor to be saved.
   */
  private Path getBackupDescriptorPath() {
    return Paths.get(upgradeConfigService.getBackupFolderName(), upgradeConfigService.getBackupDescriptorFileName());
  }

  /**
   * This function deletes the backup directory.
   *
   * @param muleHome location of the backup directory.
   * @throws IOException
   */
  private void deleteBackupFolder(Path muleHome, boolean isDryRunMode) throws IOException {
    File backupDir = muleHome.resolve(upgradeConfigService.getBackupFolderName()).toFile();
    if (backupDir.exists() && !isDryRunMode) {
      FileUtils.forceDelete(backupDir);
    }
  }

  public Clock getClock() {
    if (clock == null) {
      clock = Clock.systemDefaultZone();
    }
    return clock;
  }

  public void setClock(Clock clock) {
    this.clock = clock;
  }
}
