/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import com.mulesoft.runtime.upgrade.tool.domain.CommandOptions;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.domain.enums.PathToBeExcluded;
import com.mulesoft.runtime.upgrade.tool.domain.enums.PathToBeReplaced;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.DescriptorService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.StatusPreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeService;
import com.mulesoft.runtime.upgrade.tool.service.api.UserInteractionService;
import com.mulesoft.runtime.upgrade.tool.service.api.YamlService;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * Service to upgrade a Mule Runtime distribution.
 */
@Service
public class DefaultUpgradeService implements UpgradeService {

  public static final String USER_REJECTED_TO_PROCEED_WITH_THE_UPGRADE_PROCESS_MESSAGE =
      "User has rejected to proceed with the upgrade process";

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultUpgradeService.class);

  /**
   * Upgrade tool jar is assumed to be located at /mule-home/tools directory
   */
  public static final Path DEFAULT_NEW_MULE_DISTRO_PATH = Paths.get("..");

  @Autowired
  private MuleDistroService distroService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private BackupService backupService;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  @Autowired
  private StatusPreconditionsValidatorService statusPreconditionsValidatorService;

  @Autowired
  private UserInteractionService userInteractionService;

  @Autowired
  private DescriptorService descriptorService;

  @Autowired
  private YamlService yamlService;

  /**
   * Upgrades a Mule Runtime distribution based on a new given one. The upgrade process fails if any of the preconditions are not
   * met. Depending on the given options, destructive operations could be priorly confirmed interactively by the user.
   *
   * @param oldMule        path to the Mule distribution to upgrade.
   * @param newMule        path of the Mule distribution that will be used for the upgrade.
   * @param commandOptions configuration options for the upgrade.
   * @throws IOException if any I/O error occurs.
   */
  @Override
  public void upgrade(Path oldMule, Path newMule, CommandOptions commandOptions) throws IOException {
    LOGGER.debug("Upgrading distro at [{}] from distro at [{}] ...", oldMule, newMule);

    MuleDistribution oldDistro = distroService.initMuleDistro(oldMule);
    MuleDistribution newDistro = distroService.initMuleDistro(getNewMuleDistroPath(newMule));

    LOGGER.info("Upgrading distribution at path [{}] with version: [{}] to distribution at path [{}] with version [{}] ...",
                oldDistro.getLocation(), oldDistro.getVersion(), newDistro.getLocation(), newDistro.getVersion());

    statusPreconditionsValidatorService.checkUpgradePreconditions(oldDistro, newDistro, commandOptions.isForce());
    requestUsersConfirmation(commandOptions.isForce());

    descriptorService.compareMuleDistros(oldMule);

    backupService.create(oldDistro, commandOptions.isDryRun());

    distroService.checkClusterModeIsActive(oldMule);

    // @todo[DEL-3316]: Fix bug: Patches are not being deleted ¯\_(ツ)_/¯ yet...
    copyFiles(oldMule, newMule, commandOptions.isDryRun());

    LOGGER.info("Upgrade process finished successfully.");
  }

  protected Path getNewMuleDistroPath(Path newMule) {
    if (newMule == null) {
      LOGGER.debug("Default new distro path will be used for the upgrade: {}", DEFAULT_NEW_MULE_DISTRO_PATH);
      return DEFAULT_NEW_MULE_DISTRO_PATH;
    }
    return newMule;
  }

  private void requestUsersConfirmation(boolean force) throws IOException {
    if (!force
        && !userInteractionService.confirmAction("perform the upgrade", "Do you want to proceed with the upgrade?", false)) {
      throw new IllegalStateException(USER_REJECTED_TO_PROCEED_WITH_THE_UPGRADE_PROCESS_MESSAGE);
    }
  }

  private void copyFiles(Path oldMule, Path newMule, boolean isDryRunMode) throws IOException {
    try {
      LOGGER.debug("Copying the new files...");
      fileSystemService.copyFiles(newMule, oldMule, PathToBeReplaced.getAllPaths(), PathToBeExcluded.getAllPaths(), isDryRunMode);
    } catch (IOException e) {
      throw new IOException("Failed to copy files from newer Mule Runtime distribution. Consider a rollback after having " +
          "solved the underlying failure condition, if any.", e);
    }
  }
}
