/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.command;

import static picocli.CommandLine.Command;

import com.mulesoft.runtime.upgrade.tool.domain.CommandOptions;
import com.mulesoft.runtime.upgrade.tool.service.api.RollbackService;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import picocli.CommandLine.Option;

/**
 * This class implements the rollback command received by command line.
 */
@Component
@Command(name = "rollback", description = "Rollbacks a Mule Runtime upgrade from a backup")
public class RollbackCommand implements Callable<Integer> {

  private static final Logger LOGGER = LoggerFactory.getLogger(RollbackCommand.class);

  @Option(names = {"-o", "--old-mule-distro"}, description = "File path of Mule Runtime distribution upgraded with the tool")
  Path oldMuleDistro;

  @Option(names = {"-d", "--dry-run"},
      description = "Simulate rollback process without performing any permanent changes to the current Mule Runtime Distribution.")
  boolean dryRun;

  @Option(names = {"-f", "--force"}, description = "Performs backup restoration without requesting user confirmation")
  boolean force;

  @Option(names = {"-h", "--help"}, usageHelp = true, description = "Display this help message")
  boolean usageHelpRequested;

  @Autowired
  private RollbackService rollBackService;

  public Integer call() throws IOException {
    LOGGER.debug("Mule distribution location: [{}]", oldMuleDistro);

    if (oldMuleDistro == null) {
      if (!guessMuleHome()) {
        LOGGER.error("Unable to guess a MULE_HOME candidate to be upgraded, use -o to specify a valid MULE_HOME");
        System.exit(1);
      }
    }

    CommandOptions commandOptions = CommandOptions.builder().force(force).dryRun(dryRun).build();
    rollBackService.rollback(oldMuleDistro, commandOptions);
    return 0;
  }

  public void setOldMuleDistro(Path oldMuleDistro) {
    this.oldMuleDistro = oldMuleDistro;
  }

  public boolean guessMuleHome() {
    String userDir = System.getProperty("user.dir");

    if (userDir.endsWith("tools")) {
      oldMuleDistro = Paths.get(userDir).getParent();

      return true;
    } else {
      return false;
    }
  }
}
