/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.runtime.upgrade.tool.service;

import com.mulesoft.runtime.upgrade.tool.domain.AssemblyEntry;
import com.mulesoft.runtime.upgrade.tool.domain.enums.AdditionalPathToBeExcluded;
import com.mulesoft.runtime.upgrade.tool.domain.enums.FileToBeExcluded;
import com.mulesoft.runtime.upgrade.tool.domain.enums.PathToBeReplaced;
import com.mulesoft.runtime.upgrade.tool.service.api.DescriptorService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.YamlService;
import com.mulesoft.runtime.upgrade.tool.utils.ClassLoaderService;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.CollectionType;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import org.apache.commons.io.FileUtils;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.stereotype.Service;

/**
 * This service manages file descriptor's functionalities.
 */
@Service
public class DefaultDescriptorService implements DescriptorService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDescriptorService.class);

  @Autowired
  private YamlService yamlService;

  @Autowired
  private MuleDistroService muleDistroService;

  @Autowired
  private ClassLoaderService classLoaderService;

  private static final String FILE_DESCRIPTOR_DIRECTORY_INSIDE_UPGRADE_TOOL = "descriptors/";
  private static final String FILE_DESCRIPTOR_GENERATED = "current-file-descriptor.yaml";
  private static final String FILE_DESCRIPTOR_PATH_INSIDE_MULE_DISTRIBUTION = "lib/mule/mule-assembly-content-descriptor-";
  private static final String FILE_DESCRIPTOR_DISTRO_FILE_NAME = "assembly-descriptor.yaml";
  private static final String ORIGINAL_FILE_DESCRIPTOR_INSIDE_CURRENT_DISTRO = "file-descriptor-inside-current-distro.yaml";
  public static final String LIC_FILE_EXTENSION_TO_REMOVE = "lic";
  private static final String MULE_UPGRADE_TOOL_PATTERN_STRING =
      "(.+[/\\\\])?mule-runtime-upgrade-tool-((?:\\d+\\.){2}\\d+(?:-\\w+)?)\\.jar$";
  private static final String FILE_NOT_FOUND_EXCEPTION_MESSAGE = "Upgrade Tool doesn't have file descriptor of this version.";
  private static final Pattern MULE_UPGRADE_TOOL_PATTERN = Pattern.compile(MULE_UPGRADE_TOOL_PATTERN_STRING);

  private static final ObjectMapper MAPPER = new ObjectMapper(new YAMLFactory());
  private static final CollectionType LIST_TYPE =
      MAPPER.getTypeFactory().constructCollectionType(LinkedList.class, AssemblyEntry.class);

  private List<AssemblyEntry> entries = new ArrayList<>();

  /**
   * This method creates file descriptor of the current mule version and then compares with the newest file descriptor version.
   *
   * @param oldMulePath Path of the current mule distro.
   * @throws Exception I/O Exception.
   */
  public void compareMuleDistros(Path oldMulePath) throws IOException {
    LOGGER.debug("Creating file descriptor of old distro...");
    writeDescriptorInMuleDistro(oldMulePath);
    File file = new File(FILE_DESCRIPTOR_GENERATED);
    file.deleteOnExit();
    compare(oldMulePath);
  }

  public void compare(Path oldMulePath) throws IOException {

    List<AssemblyEntry> currentDistroFileDescriptor;
    List<AssemblyEntry> originalFileDescriptor;

    String oldMuleVersion = muleDistroService.detectMuleVersion(oldMulePath);

    originalFileDescriptor = getAssemblyEntryListByFileDescriptor(oldMuleVersion, oldMulePath);

    if (originalFileDescriptor.isEmpty())
      throw new FileNotFoundException("The File Descriptor wasn't found.");

    LOGGER.debug("Comparing file descriptors...");

    List<String> excludedPaths = getListOfExcludedPaths();

    List<AssemblyEntry> originalFileDescriptorWithOutParentDirectory = originalFileDescriptor.stream().map(assemblyEntry -> {
      List<String> pathsElements = new ArrayList<>(Arrays.asList(assemblyEntry.getName().split("/")));
      pathsElements.remove(0);
      String newAssemblyEntryName = String.join("/", pathsElements);
      return new AssemblyEntry(newAssemblyEntryName, assemblyEntry.getSizeInBytes(), assemblyEntry.getSha256());
    }).collect(Collectors.toList());

    currentDistroFileDescriptor = getAssemblyEntryListOfFileDescriptorFromGeneratedFileDescriptor(oldMulePath);

    compareFileDescriptors(currentDistroFileDescriptor, originalFileDescriptorWithOutParentDirectory, excludedPaths);
  }

  /**
   * Looking for File Descriptor inside the tool, and if doesn't exist there, try to find it inside the Mule Runtime Distribution.
   * 
   * @param currentVersion Mule Runtime Distribution version.
   * @param mulePath       Mule Runtime Distribution Path.
   * @return Assembly Entries List.
   */
  public List<AssemblyEntry> getAssemblyEntryListByFileDescriptor(String currentVersion, Path mulePath) {
    List<AssemblyEntry> assemblyEntryList = new LinkedList<>();
    try {
      assemblyEntryList =
          getAssemblyEntryListOfFileDescriptorFromUpgradeTool(currentVersion);
    } catch (IOException ex) {
      LOGGER.debug(ex.getMessage());
      try {
        assemblyEntryList = getAssemblyEntryListOfFileDescriptorFromCurrentDistribution(currentVersion, mulePath);
      } catch (IOException exception) {
        LOGGER.debug(ex.getMessage());
      }
    }
    return assemblyEntryList;
  }

  List<AssemblyEntry> getAssemblyEntryListOfFileDescriptorFromUpgradeTool(String currentVersion) throws IOException {
    List<AssemblyEntry> assemblyEntries;
    Resource descriptor =
        classLoaderService.getResourceFromClasspath(FILE_DESCRIPTOR_DIRECTORY_INSIDE_UPGRADE_TOOL + currentVersion + ".yaml");
    if (descriptor != null && descriptor.exists()) {
      assemblyEntries = MAPPER.readValue(descriptor.getInputStream(), LIST_TYPE);
    } else {
      throw new FileNotFoundException(FILE_NOT_FOUND_EXCEPTION_MESSAGE);
    }
    return assemblyEntries;
  }

  List<AssemblyEntry> getAssemblyEntryListOfFileDescriptorFromGeneratedFileDescriptor(Path oldMulePath) throws IOException {
    return MAPPER.readValue(new File(getDescriptorPathInDistro(oldMulePath).toString()), LIST_TYPE);
  }

  List<AssemblyEntry> getAssemblyEntryListOfFileDescriptorFromCurrentDistribution(String currentVersion, Path oldMulePath)
      throws IOException {
    URL url =
        new URL("jar:file:" + oldMulePath.resolve(FILE_DESCRIPTOR_PATH_INSIDE_MULE_DISTRIBUTION + currentVersion + ".jar") + "!/"
            + FILE_DESCRIPTOR_DISTRO_FILE_NAME);

    JarURLConnection connection = (JarURLConnection) url.openConnection();
    File fileDescriptor = new File(ORIGINAL_FILE_DESCRIPTOR_INSIDE_CURRENT_DISTRO);
    fileDescriptor.deleteOnExit();
    InputStream inputStreamFile = connection.getInputStream();

    try {
      FileUtils.copyInputStreamToFile(inputStreamFile, fileDescriptor);
      return MAPPER.readValue(fileDescriptor, LIST_TYPE);

    } catch (IOException ex) {
      throw ex;
    } finally {
      inputStreamFile.close();
    }
  }

  public void compareFileDescriptors(List<AssemblyEntry> currentDistroFileDescriptor, List<AssemblyEntry> originalFileDescriptor,
                                     List<String> excludedPaths) {
    currentDistroFileDescriptor.forEach(entry -> {
      if (!originalFileDescriptor.contains(entry)
          && excludedPaths.stream().noneMatch(excludedPath -> pathMatches(entry, excludedPath))) {
        LOGGER.warn("Unknown file {} was found in the runtime installation, and will be kept.", entry.getName());
      }
    });

  }

  /**
   * Makes a list with the names of all files which appears in the File Descriptor of an specific Mule Runtime Distribution.
   * 
   * @param muleDistroPath Mule Runtime Distribution Path.
   * @return List of file names.
   * @throws IOException
   */
  public List<String> getFilesNamesList(Path muleDistroPath) throws IOException {
    String currentVersion = muleDistroService.detectMuleVersion(muleDistroPath);
    List<AssemblyEntry> currentDistroAssemblyEntryList =
        getAssemblyEntryListByFileDescriptor(currentVersion, muleDistroPath);

    return currentDistroAssemblyEntryList.stream().map(entry -> {
      List<String> entryList = new ArrayList<>(Arrays.asList(entry.getName().split("/")));
      entryList.remove(0);
      return String.join("/", entryList);
    }).collect(Collectors.toList());
  }


  void logEntriesNotPresentInOldDistroToBeDeleted(List<AssemblyEntry> oldDistroFileDescriptor,
                                                  List<AssemblyEntry> newDistroFileDescriptor) {
    oldDistroFileDescriptor.forEach(oldDistroEntry -> {
      if (!newDistroFileDescriptor.contains(oldDistroEntry)) {
        LOGGER.warn("Unknown file {} was found in the runtime installation, and will be kept.", oldDistroEntry.getName());
      }
    });
  }

  List<AssemblyEntry> readDescriptorFromMuleDistro(Path distroPath) throws IOException {
    Path fileDescriptorInDistro = getDescriptorPathInDistro(distroPath);
    return readDescriptor(fileDescriptorInDistro);
  }

  Path getDescriptorPathInDistro(Path distroPath) {
    return Paths.get(distroPath.toAbsolutePath().toString(), FILE_DESCRIPTOR_GENERATED);
  }

  List<AssemblyEntry> readDescriptor(Path fileDescriptor) throws IOException {
    ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
    CollectionType listType = mapper.getTypeFactory().constructCollectionType(LinkedList.class, AssemblyEntry.class);

    return mapper.readValue(fileDescriptor.toFile(), listType);
  }

  List<AssemblyEntry> removeIgnoredEntries(List<AssemblyEntry> assemblyEntries) {
    List<AssemblyEntry> descriptorWoutIgnoredEntries;
    descriptorWoutIgnoredEntries = removeExcludedPaths(assemblyEntries);
    descriptorWoutIgnoredEntries = removeUpgradeToolEntry(descriptorWoutIgnoredEntries);
    descriptorWoutIgnoredEntries = removeEntriesWithFileExtension(descriptorWoutIgnoredEntries, LIC_FILE_EXTENSION_TO_REMOVE);
    return descriptorWoutIgnoredEntries;
  }

  List<AssemblyEntry> removeEntriesWithFileExtension(List<AssemblyEntry> descriptorEntries, String fileExtensionToRemove) {
    List<AssemblyEntry> entriesWithoutFileExtensionToRemove = new ArrayList<>(descriptorEntries);
    entriesWithoutFileExtensionToRemove
        .removeIf(entry -> FilenameUtils.getExtension(entry.getName()).equals(fileExtensionToRemove));
    return entriesWithoutFileExtensionToRemove;
  }

  List<AssemblyEntry> removeUpgradeToolEntry(List<AssemblyEntry> descriptorEntries) {
    List<AssemblyEntry> descriptorEntriesWoutUpgradeToolEntry = new ArrayList<>(descriptorEntries);
    descriptorEntriesWoutUpgradeToolEntry.removeIf(this::entryNameMatchesUpgradeToolPattern);
    return descriptorEntriesWoutUpgradeToolEntry;
  }

  private boolean entryNameMatchesUpgradeToolPattern(AssemblyEntry upgradeToolEntry) {
    return MULE_UPGRADE_TOOL_PATTERN.matcher(upgradeToolEntry.getName()).matches();
  }

  List<AssemblyEntry> removeExcludedPaths(List<AssemblyEntry> descriptorEntries) {
    return removeExcludedPaths(descriptorEntries, getListOfExcludedPaths());
  }

  List<AssemblyEntry> removeExcludedPaths(List<AssemblyEntry> descriptorEntries, List<String> excludedPaths) {
    List<AssemblyEntry> descriptorEntriesWoutExcludedPaths = new ArrayList<>(descriptorEntries);
    descriptorEntriesWoutExcludedPaths
        .removeIf(entry -> excludedPaths.stream()
            .anyMatch(excludedPath -> pathMatches(entry, excludedPath)));
    return descriptorEntriesWoutExcludedPaths;
  }

  boolean pathMatches(AssemblyEntry path, String prefix) {
    String source = path.getName().replace("/", "|").replace("\\", "|");
    String target = prefix.replace("/", "|").replace("\\", "|").replaceAll("\\.$", "");
    return source.startsWith(target);
  }

  List<String> getListOfExcludedPaths() {
    List<String> excludedPaths =
        FileToBeExcluded.getAllPaths().stream().map(Path::toString).collect(Collectors.toList());
    excludedPaths.addAll(AdditionalPathToBeExcluded.getAllPaths().stream().map(Path::toString).collect(Collectors.toList()));
    return excludedPaths;
  }

  public List<AssemblyEntry> getEntriesInPathForSelectedPaths(Path mulePath, List<Path> pathsToGetEntries) throws IOException {
    List<AssemblyEntry> currentDistroEntries = new ArrayList<>();

    for (Path path : pathsToGetEntries) {
      try {
        currentDistroEntries.addAll(getEntriesInPath(mulePath.resolve(path)));
      } catch (IOException e) {
        throw new IOException(path + " was not found.", e);
      }
    }
    return currentDistroEntries;
  }

  /**
   * Checks File Descriptor existence inside a specific mule distro and inside the tool.
   *
   * @param mulePath Mule Runtime Distribution path.
   * @return True if exists, false if doesn't exist.
   * @throws FileNotFoundException
   */
  public boolean checkFileDescriptorExistenceInsideMuleDistroOrInsideUpgradeTool(Path mulePath) throws IOException {
    String currentVersion = muleDistroService.detectMuleVersion(mulePath);
    File fileDescriptorInsideDistro =
        new File(String.valueOf(mulePath.resolve(FILE_DESCRIPTOR_PATH_INSIDE_MULE_DISTRIBUTION + currentVersion + ".jar")));
    Resource fileDescriptorInsideTool =
        classLoaderService.getResourceFromClasspath(FILE_DESCRIPTOR_DIRECTORY_INSIDE_UPGRADE_TOOL + currentVersion + ".yaml");
    return (fileDescriptorInsideDistro.exists() || (fileDescriptorInsideTool != null && fileDescriptorInsideTool.exists()));
  }

  @Override
  public List<AssemblyEntry> getEntriesInPath(Path directory) throws IOException {
    String parentDirectory = directory.toFile().getName();
    return recursiveFileTree(directory, parentDirectory);
  }

  @Override
  public boolean checkIntegrity(List<AssemblyEntry> descriptorEntries, Path dirWithFiles) throws IOException {
    return checkIntegrity(descriptorEntries, dirWithFiles, getListOfExcludedPaths());
  }

  private boolean checkIntegrity(List<AssemblyEntry> descriptorEntries, Path dirWithFiles, List<String> pathsToExclude)
      throws IOException {
    List<AssemblyEntry> descriptorEntriesWoutExcludedPaths = removeExcludedPaths(descriptorEntries, pathsToExclude);
    List<AssemblyEntry> actualDirWithFilesEntries = removeExcludedPaths(getEntriesInPath(dirWithFiles), pathsToExclude);
    List<AssemblyEntry> missingAndWrongHashEntriesFromDescriptor =
        filterMissingAndWrongHashEntries(descriptorEntriesWoutExcludedPaths, actualDirWithFilesEntries);
    List<AssemblyEntry> missingEntriesFromDescriptor =
        filterMissingEntries(descriptorEntriesWoutExcludedPaths, actualDirWithFilesEntries);
    List<AssemblyEntry> wrongHashEntriesFromDescriptor =
        getWrongHashEntries(missingAndWrongHashEntriesFromDescriptor, missingEntriesFromDescriptor);

    List<AssemblyEntry> extraEntriesInActualDir =
        filterMissingEntries(actualDirWithFilesEntries, descriptorEntriesWoutExcludedPaths);
    logMissingEntries(missingEntriesFromDescriptor, dirWithFiles);
    logExtraEntries(extraEntriesInActualDir, dirWithFiles);
    logEntriesWithWrongHash(wrongHashEntriesFromDescriptor, actualDirWithFilesEntries, dirWithFiles);

    return areEntriesWithIntegrityErrors(missingAndWrongHashEntriesFromDescriptor, extraEntriesInActualDir);
  }

  private List<AssemblyEntry> getWrongHashEntries(List<AssemblyEntry> missingAndWrongHashEntriesFromDescriptor,
                                                  List<AssemblyEntry> missingEntriesFromDescriptor) {
    List<AssemblyEntry> wrongHashEntriesFromDescriptor = new ArrayList<>(missingAndWrongHashEntriesFromDescriptor);
    wrongHashEntriesFromDescriptor.removeAll(missingEntriesFromDescriptor);
    return wrongHashEntriesFromDescriptor;
  }

  private boolean areEntriesWithIntegrityErrors(List<AssemblyEntry> missingAndWrongHashEntriesFromDescriptor,
                                                List<AssemblyEntry> extraEntriesInActualDir) {
    return missingAndWrongHashEntriesFromDescriptor.isEmpty() && extraEntriesInActualDir.isEmpty();
  }

  private void logEntriesWithWrongHash(List<AssemblyEntry> wrongHashEntries, List<AssemblyEntry> actualDirWithFilesEntries,
                                       Path dirWithFiles) {
    for (AssemblyEntry wrongHashEntry : wrongHashEntries) {
      Optional<AssemblyEntry> entryFoundInActualDir = findByEntryName(actualDirWithFilesEntries, wrongHashEntry);
      String entryFoundInActualDirHash =
          (entryFoundInActualDir.isPresent()) ? entryFoundInActualDir.get().getSha256() : "<problem getting actual hash>";
      LOGGER.warn("{} with expected hash {} doesn't match with the hash {} of the file in folder {}.", wrongHashEntry.getName(),
                  wrongHashEntry.getSha256(), entryFoundInActualDirHash, dirWithFiles);
    }
  }

  private void logMissingEntries(List<AssemblyEntry> missingEntries, Path dirWithFiles) {
    missingEntries
        .forEach(missingEntry -> LOGGER.warn("{} file wasn't found in folder {}.", missingEntry.getName(), dirWithFiles));
  }

  private void logExtraEntries(List<AssemblyEntry> extraEntries, Path dirWithFiles) {
    extraEntries.forEach(extraEntry -> LOGGER.warn("{} extra file was found in folder {}.", extraEntry.getName(), dirWithFiles));
  }

  private List<AssemblyEntry> filterMissingAndWrongHashEntries(List<AssemblyEntry> entriesOne, List<AssemblyEntry> entriesTwo) {
    return entriesOne.stream().filter(entryOne -> !entriesTwo.contains(entryOne))
        .collect(Collectors.toList());
  }

  private List<AssemblyEntry> filterMissingEntries(List<AssemblyEntry> entriesOne, List<AssemblyEntry> entriesTwo) {
    return entriesOne.stream().filter(entryOne -> !containsEntryByName(entriesTwo, entryOne))
        .collect(Collectors.toList());

  }

  private boolean containsEntryByName(List<AssemblyEntry> entries, AssemblyEntry entryToFindByName) {
    return entries.stream().anyMatch(entry -> entry.getName().equals(entryToFindByName.getName()));
  }

  private Optional<AssemblyEntry> findByEntryName(List<AssemblyEntry> entries, AssemblyEntry entryToFind) {
    return entries.stream().filter(entry -> entry.getName().equals(entryToFind.getName())).findFirst();
  }

  void writeDescriptorInMuleDistro(Path muleDistroPath) throws IOException {
    List<AssemblyEntry> currentDistroEntries = getEntriesInPathForSelectedPaths(muleDistroPath, PathToBeReplaced.getAllPaths());
    writeFileDescriptor(currentDistroEntries, getDescriptorPathInDistro(muleDistroPath));
  }

  void writeFileDescriptor(List<AssemblyEntry> currentDistroEntries, Path fileDescriptorPath) throws IOException {
    try {
      yamlService.writeValueToFile(fileDescriptorPath.toFile(), currentDistroEntries);
    } catch (IOException e) {
      throw new IOException("Error generating the distribution assembly content descriptor", e);
    }
  }

  AssemblyEntry extractEntryInfo(Path entryPath, String parentFoldersFromBase) throws IOException {
    if (Files.isDirectory(entryPath)) {
      throw new IllegalArgumentException("Error generating the Entry for file " + entryPath.toAbsolutePath()
          + " because it is a directory.");
    }

    try {
      byte[] contents = FileUtils.readFileToByteArray(entryPath.toFile());
      String sha256 = DigestUtils.sha256Hex(contents);
      return new AssemblyEntry(parentFoldersFromBase + "/" + entryPath.getFileName().toString(), Files.size(entryPath), sha256);
    } catch (IOException e) {
      throw new IOException("Error generating the Entry info for file " + entryPath.toAbsolutePath(), e);
    }
  }

  private List<AssemblyEntry> recursiveFileTree(Path path, String parentFoldersFromBase) throws IOException {

    List<AssemblyEntry> assemblyEntries = new ArrayList<>();

    File file = path.toFile();

    if (file.isFile()) {
      assemblyEntries.add(extractEntryInfo(file.toPath(), parentFoldersFromBase));
    }
    if (file.isDirectory()) {
      String parentAux = parentFoldersFromBase + "/" + file.getName();
      if (parentFoldersFromBase.equals(file.getName())) {
        parentAux = parentFoldersFromBase;
      }

      for (File elem : file.listFiles()) {
        assemblyEntries.addAll(recursiveFileTree(elem.toPath(), parentAux));
      }
    }
    return assemblyEntries;
  }
}
