/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static java.lang.String.format;
import static java.util.regex.Pattern.compile;

import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.service.api.ConfigFilesService;
import com.mulesoft.runtime.upgrade.tool.service.api.DescriptorService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleStatusService;
import com.mulesoft.runtime.upgrade.tool.utils.ClassLoaderService;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.NotDirectoryException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.filefilter.RegexFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A service for {@link MuleDistribution} operations.
 */
@Service
public class DefaultMuleDistroService implements MuleDistroService {

  private static final Pattern GET_VERSION_MULE_DIRNAME_PATTERN = Pattern.compile("4\\.(\\d+)\\.(\\d+)");

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultMuleDistroService.class);

  private static final String MULE_CLUSTER_PROPERTIES_INNER_PATH = ".mule/mule-cluster.properties";

  public static final Path LIB_MULE_JAR_DIRECTORY_PATH = Paths.get("lib", "mule");

  public static final Path CONFIG_FOLDER = Paths.get("conf");

  // Regex specially crafted to prevent Regular expression Denial of Service - ReDoS
  private static final Pattern MULE_CORE_JAR_FILE_NAME_PATTERN = compile("^mule-core-((?:\\d+\\.){2}\\d+(?:-\\w+)?)\\.jar$");

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private MuleStatusService muleStatusService;

  @Autowired
  private ClassLoaderService classLoaderService;

  @Autowired
  private DescriptorService descriptorService;

  @Autowired
  private MuleDistroService distroService;

  @Autowired
  private ConfigFilesService configFilesService;

  /**
   * Initializes a {@link MuleDistribution} object after validating that the given file path corresponds to a Mule Runtime
   * distribution.
   *
   * @param muleHome a path pointing to a Mule Runtime distribution.
   * @return a {@link MuleDistribution} instance.
   * @throws NotDirectoryException if given muleHome path is not a directory.
   * @throws FileNotFoundException if mule-core jar used for determining the Mule distro version is not found.
   */
  @Override
  public MuleDistribution initMuleDistro(Path muleHome) throws IOException {
    LOGGER.debug("Initializing Mule runtime distro object for location [{}]", muleHome);
    fileSystemService.checkIsADirectory(muleHome);
    String muleVersion = detectMuleVersion(muleHome);
    return new MuleDistribution.Builder().location(muleHome).version(muleVersion).build();
  }

  /**
   * Introspects the given directory to determine if it looks like a Mule Runtime distribution.
   *
   * @param muleDistroLocation a path pointing to a potential Mule Runtime distribution.
   * @throws IOException if given muleHome path does not look like a distro.
   */
  @Override
  public void looksLikeADistro(Path muleDistroLocation) throws IOException {
    fileSystemService.checkIsADirectory(muleDistroLocation);
    detectMuleVersion(muleDistroLocation);
  }

  /**
   * Checks if current mule distro is working on cluster mode.
   *
   * @param currentMuleDistroPropertiesFilePath Current mule distro path.
   * @throws IOException I/O Exception.
   * @return
   */
  public boolean checkClusterModeIsActive(Path currentMuleDistroPropertiesFilePath) throws IOException {
    LOGGER.debug("Checking if cluster mode is active...");
    String currentMuleDistroPropertiesFile =
        currentMuleDistroPropertiesFilePath.resolve(MULE_CLUSTER_PROPERTIES_INNER_PATH).toString();
    File file = new File(currentMuleDistroPropertiesFile);

    if (isClusterModeActive(file, currentMuleDistroPropertiesFile)) {
      LOGGER
          .warn("A cluster configuration was found in this Mule Runtime installation. You will need to execute the" +
              " upgrade process for every node in the cluster.");
      return true;
    }
    return false;
  }

  /**
   * Checks if a clusterId exists in Mule distribution configuration.
   *
   * @param propertiesFile                  a cluster properties file.
   * @param currentMuleDistroPropertiesFile a path to the cluster properties file.
   * @return true if cluster mode is active, false otherwise
   * @throws IOException
   */
  public boolean isClusterModeActive(File propertiesFile, String currentMuleDistroPropertiesFile)
      throws IOException {
    String muleClusterId;
    if (propertiesFile.exists()) {
      try (FileInputStream fileInputStream = new FileInputStream(currentMuleDistroPropertiesFile)) {
        Properties myPropertyFile = new Properties();
        myPropertyFile.load(fileInputStream);
        muleClusterId = myPropertyFile.getProperty("mule.clusterId");
      }
      return muleClusterId != null && !muleClusterId.isEmpty();
    }
    return false;
  }

  /**
   * Detects Mule Runtime Distribution version.
   *
   * @param muleHome Mule Runtime Distribution path.
   * @return Mule Runtime Distribution version.
   * @throws FileNotFoundException if muleHome path is not found.
   */
  public String detectMuleVersion(Path muleHome) throws FileNotFoundException {
    LOGGER.debug("Detecting Mule Runtime version for distribution located at: [{}]", muleHome);

    File muleCoreJarDirectory = checkLibMuleJarDirectoryExists(muleHome);

    File[] files = findFilesByRegex(muleCoreJarDirectory, MULE_CORE_JAR_FILE_NAME_PATTERN);

    if (files.length != 1) {
      throw new FileNotFoundException(format("Mule Core jar artifact does not exists at the expected location [%s]",
                                             muleCoreJarDirectory));
    } else {
      Matcher matcher = MULE_CORE_JAR_FILE_NAME_PATTERN.matcher(files[0].getName());
      matcher.find();
      String muleVersion = matcher.group(1);
      LOGGER.debug("Detected mule version for distro at [{}] is: [{}]", muleHome, muleVersion);
      return muleVersion;
    }
  }

  private File[] findFilesByRegex(File muleCoreJarDirectory, Pattern patternFileFilter) {
    FileFilter fileFilter = new RegexFileFilter(patternFileFilter);
    return muleCoreJarDirectory.listFiles(fileFilter);
  }

  private File checkLibMuleJarDirectoryExists(Path muleHome) throws FileNotFoundException {
    File muleCoreJarDirectory = muleHome.resolve(LIB_MULE_JAR_DIRECTORY_PATH).toFile();
    if (!muleCoreJarDirectory.exists()) {
      throw new FileNotFoundException(format("Expected dir [%s] does not exist. Given path [%s] might not correspond to an actual distro",
                                             muleCoreJarDirectory, muleHome));
    }
    return muleCoreJarDirectory;
  }

  /**
   *
   * Checks File Descriptor existence inside all mule distros involve in upgrade process and inside the tool.
   *
   * @param oldMulePath Current Mule Runtime Distribution path.
   * @param newMulePath New Mule Runtime Distribution path.
   * @return True if exist in both distros, false if doesn't exist in one of the distros.
   * @throws FileNotFoundException
   */
  @Override
  public boolean checkFilesDescriptorsExistencesInsideBothDistributions(Path oldMulePath, Path newMulePath)
      throws IOException {
    LOGGER.debug("Checking file descriptor existence...");
    return descriptorService.checkFileDescriptorExistenceInsideMuleDistroOrInsideUpgradeTool(oldMulePath)
        && descriptorService.checkFileDescriptorExistenceInsideMuleDistroOrInsideUpgradeTool(newMulePath);
  }

  @Override
  public Path getLibMuleJarDirectoryPath() {
    return LIB_MULE_JAR_DIRECTORY_PATH;
  }

  @Override
  public Path getConfigFolder() {
    return CONFIG_FOLDER;
  }

  /**
   * Check if the name of the Mule directory is misleading regarding its content
   *
   * @param currentMule Mule Runtime Distribution path.
   * @return A boolean
   */
  @Override
  public boolean checkMisleadingMuleDirName(Path currentMule) {

    String muleDirName = currentMule.getFileName().toString();
    Matcher matchMuleDirName = GET_VERSION_MULE_DIRNAME_PATTERN.matcher(muleDirName);

    if (matchMuleDirName.find()) {
      return true;
    }
    return false;
  }

}
