/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.utils;

import static org.apache.commons.io.IOUtils.toByteArray;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utils for loading resources.
 */
public class JarFileUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(JarFileUtils.class);
  private static final String META_INF_PREFIX = "META-INF/";

  private JarFileUtils() {
    // sonarqube java:S1118
    throw new IllegalStateException("Utility class");
  }

  /**
   * Get a list with the files names inside a Jar File
   * 
   * @param jarPath Jar File Path
   * @return A list of files inside a Jar File
   * @throws IOException
   */
  public static List<String> getJarEntries(String jarPath) throws IOException {
    List<String> filesInsideJar = new ArrayList<>();
    try (JarFile jarFile = new JarFile(jarPath)) {
      jarFile.stream().forEach(jarEntry -> {
        if (!jarEntry.isDirectory() && !jarEntry.getName().startsWith(META_INF_PREFIX)) {
          filesInsideJar.add(jarEntry.getName());
        }
      });
    }
    return filesInsideJar;
  }

  /**
   * Get the file content in byte[] of a file inside a jar
   *
   * @param jarPath       Jar File Path
   * @param fileInsideJar File path inside the jar
   * @return A byte[] with the content of the file
   * @throws IOException
   */
  public static byte[] getFileContentInByteArray(String jarPath, String fileInsideJar) throws IOException {
    LOGGER.debug("Getting content of file [{}] inside jar [{}]...", fileInsideJar, jarPath);
    try (JarFile jarFile = new JarFile(new File(jarPath))) {
      JarEntry fileInJarEntry = jarFile.getJarEntry(fileInsideJar);
      try (InputStream fileContent = jarFile.getInputStream(fileInJarEntry)) {
        return toByteArray(fileContent);
      }
    }
  }

}
