/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.domain;

import static java.lang.String.format;

import java.util.Comparator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A non SemVer compliant spec class, tailored to the requirements of this business domain.
 */
public class SemVer implements Comparable<SemVer> {

  private static final String SEM_VER_PARSE_REGEX = "(\\d+)(?:\\.(\\d+)(?:\\.(\\d+))?(?:\\-([\\w][\\w\\.\\-_]*))?)?";

  private int major;
  private int minor;
  private int patch;
  private String suffix;

  /**
   * Gets a {@code SemVer} value after performing a naive and lightway parsing of the given string.
   *
   * @param semVer a string representing a semVer value.
   * @return a {@code SemVer} value.
   * @throws IllegalArgumentException if given string can't be parsed.
   */
  public static SemVer valueOf(String semVer) {
    Pattern compile = Pattern.compile(SEM_VER_PARSE_REGEX);
    Matcher matcher = compile.matcher(semVer);

    if (!matcher.matches()) {
      throw new IllegalArgumentException(format("%s could not be parsed as semVer value", semVer));
    }

    String suffix = matcher.group(4) != null ? matcher.group(4) : "";
    return new SemVer(
                      Integer.valueOf(matcher.group(1)),
                      Integer.valueOf(matcher.group(2)),
                      Integer.valueOf(matcher.group(3)),
                      suffix);
  }

  private SemVer(int major, int minor, int patch, String metadata) {
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.suffix = metadata;
  }

  public int getMajor() {
    return major;
  }

  public int getMinor() {
    return minor;
  }

  public int getPatch() {
    return patch;
  }

  public String getSuffix() {
    return suffix;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (o == null || getClass() != o.getClass())
      return false;

    SemVer semVer = (SemVer) o;

    if (getMajor() != semVer.getMajor())
      return false;
    if (getMinor() != semVer.getMinor())
      return false;
    if (getPatch() != semVer.getPatch())
      return false;
    return getSuffix().equals(semVer.getSuffix());
  }

  @Override
  public int hashCode() {
    int result = getMajor();
    result = 31 * result + getMinor();
    result = 31 * result + getPatch();
    result = 31 * result + getSuffix().hashCode();
    return result;
  }

  @Override
  public String toString() {
    if (suffix.isEmpty()) {
      return String.format("%s.%s.%s", major, minor, patch);
    }
    return String.format("%s.%s.%s-%s", major, minor, patch, suffix);
  }

  @Override
  public int compareTo(SemVer o) {
    return Comparator
        .comparingInt(SemVer::getMajor)
        .thenComparingInt(SemVer::getMinor)
        .thenComparingInt(SemVer::getPatch)
        .thenComparing(SemVer::getSuffix)
        .compare(this, o);
  }
}
