/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.utils.JarFileUtils.getFileContentInByteArray;
import static com.mulesoft.runtime.upgrade.tool.utils.JarFileUtils.getJarEntries;
import static org.apache.commons.io.FileUtils.copyFile;
import static org.apache.commons.io.FileUtils.copyFileToDirectory;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.apache.commons.io.FileUtils.deleteDirectory;
import static org.apache.commons.io.FileUtils.writeByteArrayToFile;

import com.mulesoft.runtime.upgrade.tool.service.api.ConfigFilesService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;
import com.mulesoft.runtime.upgrade.tool.utils.ClassLoaderService;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

/**
 * This service manages file descriptor's functionalities.
 */
@Service
public class DefaultConfigFilesService implements ConfigFilesService {

  @Autowired
  private MuleDistroService muleDistroService;

  @Autowired
  private ClassLoaderService classLoaderService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  public static final String ORIGINAL_FILES_OLD_DISTRO_FOLDER = "ORIGINAL_CONFIG_FILES_OLD_DISTRO";
  public static final String ORIGINAL_FILES_NEW_DISTRO_FOLDER = "ORIGINAL_CONFIG_FILES_NEW_DISTRO";

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultConfigFilesService.class);
  private static final String CONFIGS_FILES_JAR_NAME_PREFIX = "mule-config-files-";
  private static final String CONFIGS_FILES_JARS_DIRECTORY_INSIDE_UPGRADE_TOOL = "mule-config-files/";
  private static final String TO_FIX_SUFFIX = "_TO_FIX";

  @Override
  public String getFilesWithConflictsSuffix() {
    return TO_FIX_SUFFIX;
  }

  @Override
  public Path getResolvedConflictsFolder() {
    return Paths.get(upgradeConfigService.getUpgradeToolFolderName(),
                     upgradeConfigService.getConfigFilesResolvedConflictsFolderName());
  }

  String getConfigsFileJarPath(Path muleHome) throws IOException {
    if (checkConfigsFileJarExistenceInsideUpgradeTool(muleHome)) {
      String version = muleDistroService.detectMuleVersion(muleHome);
      return extractConfigFilesJarFromUpgradeTool(version, muleHome);
    } else if (checkConfigsFileJarExistenceInsideMuleDistro(muleHome)) {
      return resolveConfigsFilesJarPathInDistro(muleHome);
    } else {
      throw new FileNotFoundException("The Configs Jar file was not found neither in the Mule Distribution " + muleHome
          + " nor inside the upgrade tool configs files jars.");
    }
  }

  String extractConfigFilesJarFromUpgradeTool(String version, Path muleHome) throws IOException {
    Optional<String> configFilesJarURLOptional = resolveConfigsFilesJarPathInUpgradeTool(version);

    if (configFilesJarURLOptional.isPresent()) {
      String configFilesJarInUpgradeTool = configFilesJarURLOptional.get();
      String configFilesJarName = Paths.get(configFilesJarInUpgradeTool).getFileName().toString();
      File destExtractedFile = getResolvedConflictsFolderInDistro(muleHome).resolve(configFilesJarName).toFile();
      Resource fileDescriptorInsideTool =
          classLoaderService.getResourceFromClasspath(configFilesJarInUpgradeTool);
      copyInputStreamToFile(fileDescriptorInsideTool.getInputStream(), destExtractedFile);
      return destExtractedFile.getAbsolutePath();
    } else {
      throw new FileNotFoundException("The Configs Jar file was not found inside the upgrade tool configs files jars.");
    }
  }

  boolean checkConfigsFileJarExistenceInsideUpgradeTool(Path muleHome) throws FileNotFoundException {
    String currentVersion = muleDistroService.detectMuleVersion(muleHome);
    return checkConfigsFileJarExistenceInsideUpgradeTool(currentVersion);
  }

  boolean configFileIsInResolvedConflictsFolder(Path muleHome, String fileName) {
    return Files.exists(getResolvedConflictsFolderInDistro(muleHome).resolve(fileName));
  }

  public Path getResolvedConflictsFolderInDistro(Path muleHome) {
    return muleHome.resolve(getResolvedConflictsFolder());
  }

  boolean checkConfigsFileJarExistenceInsideUpgradeTool(String version) {
    Optional<String> configsFileJarInUpgradeTool = resolveConfigsFilesJarPathInUpgradeTool(version);

    return configsFileJarInUpgradeTool.isPresent();
  }

  boolean checkConfigsFileJarExistenceInsideMuleDistro(Path muleHome) throws FileNotFoundException {
    String configsFileJarInDistro = resolveConfigsFilesJarPathInDistro(muleHome);

    if (Files.exists(Paths.get(configsFileJarInDistro))) {
      LOGGER.debug("{} file was found inside the distribution.", configsFileJarInDistro);
      return true;
    } else {
      LOGGER.debug("{} file was NOT found inside the distribution.", configsFileJarInDistro);
      return false;
    }
  }

  Optional<String> resolveConfigsFilesJarPathInUpgradeTool(String version) {
    String configFilesJarPathInUpgradeTool =
        CONFIGS_FILES_JARS_DIRECTORY_INSIDE_UPGRADE_TOOL + CONFIGS_FILES_JAR_NAME_PREFIX + version + ".jar";
    URL configsFileJarInsideUpgradeToolURL =
        classLoaderService.getURLByResourcePath(configFilesJarPathInUpgradeTool);
    if (configsFileJarInsideUpgradeToolURL == null) {
      LOGGER.debug("Mule Runtime configs files jar NOT found in upgrade tool under resources path [{}]",
                   configFilesJarPathInUpgradeTool);
      return Optional.empty();
    }
    LOGGER.debug("Mule Runtime configs files jar found in upgrade tool under resources path [{}]",
                 configFilesJarPathInUpgradeTool);
    return Optional.of(configFilesJarPathInUpgradeTool);
  }

  String resolveConfigsFilesJarPathInDistro(Path muleHome) throws FileNotFoundException {
    String currentVersion = muleDistroService.detectMuleVersion(muleHome);
    Path configsFileJarWithVersion =
        muleDistroService.getLibMuleJarDirectoryPath().resolve(CONFIGS_FILES_JAR_NAME_PREFIX + currentVersion + ".jar");
    return muleHome.resolve(configsFileJarWithVersion).toAbsolutePath().toString();
  }

  @Override
  public boolean checkConflicts(Path oldMule, Path newMule, boolean isDryRunMode) throws IOException {
    LOGGER.debug("Checking Mule Runtime Config files conflicts between distributions [{}] and [{}]...", oldMule, newMule);
    if (checkConfigFilesJarsExistencesInsideBothDistributions(oldMule, newMule)) {
      LOGGER.debug("Mule Runtime Config files successfully found in both distributions [{}] and [{}]", oldMule, newMule);
    } else {
      return false;
    }

    String configsFileJarPathInNewDistro;
    String configsFileJarPathInOldDistro;
    try {
      configsFileJarPathInNewDistro = getConfigsFileJarPath(newMule);
      configsFileJarPathInOldDistro = getConfigsFileJarPath(oldMule);
    } catch (FileNotFoundException e) {
      throw new IOException("Upgrading from/to this version caused an unexpected error trying to obtain the Mule Runtime configs files. Please create a ticket to support.",
                            e);
    }

    List<String> configFilesInOldDistroJar = getJarEntries(configsFileJarPathInOldDistro);

    boolean configFilesCheckIsFine = true;
    for (String configFileNameInOldDistroJar : configFilesInOldDistroJar) {
      byte[] configFileContentInOldDistro =
          getFileContentInByteArray(configsFileJarPathInOldDistro, configFileNameInOldDistroJar);
      byte[] configFileContentInNewDistro =
          getFileContentInByteArray(configsFileJarPathInNewDistro, configFileNameInOldDistroJar);

      if (configFileIsInResolvedConflictsFolder(oldMule, configFileNameInOldDistroJar)) {
        continue;
      }
      if (configFileHasConflicts(configFileContentInOldDistro, configFileContentInNewDistro)) {
        if (!isDryRunMode) {
          copyUserConfigFileWithConflictsAddingToFixSuffix(oldMule, configFileNameInOldDistroJar);
          copyOriginalConfigFilesInConflictsFolderAsReference(oldMule, newMule, configFileContentInOldDistro,
                                                              configFileContentInNewDistro,
                                                              configFileNameInOldDistroJar);
        }
        String errorMessage =
            "The config file '{}' was updated in the new version, make sure the one you want to use with the changes needed is in [{}] to be used during the upgrade command.";
        LOGGER.error(errorMessage, configFileNameInOldDistroJar,
                     getResolvedConflictsFolderInDistro(oldMule).resolve(configFileNameInOldDistroJar));
        configFilesCheckIsFine = false;

      } else {
        if (!isDryRunMode) {
          copyUserConfigFileWoutConflicts(oldMule, configFileNameInOldDistroJar);
        }
      }
    }

    return configFilesCheckIsFine;
  }

  private void copyUserConfigFileWoutConflicts(Path muleHome, String configFileName) throws IOException {
    File sourceFile = muleHome.resolve(configFileName).toFile();
    File destFile =
        getResolvedConflictsFolderInDistro(muleHome).resolve(muleDistroService.getConfigFolder()).toFile();

    copyFileToDirectory(sourceFile, destFile);
  }

  private void copyUserConfigFileWithConflictsAddingToFixSuffix(Path muleHome, String configFileName) throws IOException {
    File sourceFile = muleHome.resolve(configFileName).toFile();
    File destFile = getResolvedConflictsFolderInDistro(muleHome).resolve(configFileName + TO_FIX_SUFFIX).toFile();

    copyFile(sourceFile, destFile);
  }

  private void copyOriginalConfigFilesInConflictsFolderAsReference(Path muleHomeOld, Path muleHomeNew,
                                                                   byte[] oldConfigFileContent,
                                                                   byte[] newConfigFileContent, String configFileName)
      throws IOException {
    File destFileOldDistro = getResolvedConflictsFolderInDistro(muleHomeOld).resolve(
                                                                                     ORIGINAL_FILES_OLD_DISTRO_FOLDER
                                                                                         + "-"
                                                                                         + muleHomeOld
                                                                                             .getFileName())
        .resolve(configFileName).toFile();
    File destFileNewDistro = getResolvedConflictsFolderInDistro(muleHomeOld).resolve(
                                                                                     ORIGINAL_FILES_NEW_DISTRO_FOLDER
                                                                                         + "-"
                                                                                         + muleHomeNew
                                                                                             .getFileName())
        .resolve(configFileName).toFile();

    writeByteArrayToFile(destFileOldDistro, oldConfigFileContent);
    writeByteArrayToFile(destFileNewDistro, newConfigFileContent);
  }

  private boolean configFileHasConflicts(byte[] configFileContentInOldDistro, byte[] configFileContentInNewDistro)
      throws IOException {
    return !fileSystemService.areHashesBetweenFilesEqual(configFileContentInOldDistro, configFileContentInNewDistro);
  }

  /**
   * Checks File Descriptor existence inside all mule distros involve in upgrade process and inside the tool.
   *
   * @param oldMulePath Current Mule Runtime Distribution path.
   * @param newMulePath New Mule Runtime Distribution path.
   * @return True if exist in both distros, false if doesn't exist in one of the distros.
   * @throws FileNotFoundException if the config file jar is not found in both the upgrade tool and the distro in the old or new
   *                               distros
   */
  boolean checkConfigFilesJarsExistencesInsideBothDistributions(Path oldMulePath, Path newMulePath)
      throws FileNotFoundException {
    LOGGER.debug("Checking mule-config-files jar existence...");
    return checkConfigsFileJarExistenceInsideMuleDistroOrInsideUpgradeTool(oldMulePath)
        && checkConfigsFileJarExistenceInsideMuleDistroOrInsideUpgradeTool(newMulePath);
  }

  /**
   * Checks File Descriptor existence inside a specific mule distro and inside the tool.
   *
   * @param muleHome Mule Runtime Distribution path.
   * @return True if exists, false if doesn't exist.
   * @throws FileNotFoundException if the config file jar is not found in both the upgrade tool and the distro
   */
  boolean checkConfigsFileJarExistenceInsideMuleDistroOrInsideUpgradeTool(Path muleHome)
      throws FileNotFoundException {
    return checkConfigsFileJarExistenceInsideMuleDistro(muleHome) || checkConfigsFileJarExistenceInsideUpgradeTool(muleHome);
  }

  @Override
  public void copyFinalConfigFiles(Path oldMule, boolean isDryRunMode) throws IOException {
    Path resolvedConflictsFolder =
        getResolvedConflictsFolderInDistro(oldMule);

    LOGGER.debug("Copying mule runtime distribution config files from [{}] to [{}].",
                 resolvedConflictsFolder.resolve(muleDistroService.getConfigFolder()),
                 oldMule.resolve(muleDistroService.getConfigFolder()));

    List<Path> includedPaths = new ArrayList<>();
    includedPaths.add(muleDistroService.getConfigFolder());
    List<Path> excludedPaths = new ArrayList<>();

    fileSystemService.copyFiles(resolvedConflictsFolder, oldMule, includedPaths, excludedPaths, isDryRunMode);

    if (!isDryRunMode) {
      deleteDirectory(resolvedConflictsFolder.toFile());
    }
  }
}
