/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Files.delete;
import static java.nio.file.Files.deleteIfExists;
import static java.nio.file.Files.isReadable;
import static java.nio.file.Files.isWritable;
import static java.nio.file.Files.walk;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.byteCountToDisplaySize;
import static org.apache.commons.io.FileUtils.copyDirectory;
import static org.apache.commons.io.FileUtils.deleteDirectory;
import static org.apache.commons.io.FileUtils.sizeOf;
import static org.apache.commons.io.FilenameUtils.getExtension;

import com.mulesoft.runtime.upgrade.tool.domain.enums.FileToBeExcluded;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.Files;
import java.nio.file.NotDirectoryException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A service for file system operations required for the upgrade or backup.
 */
@Service
public class DefaultFileSystemService implements FileSystemService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultFileSystemService.class);

  public static final double REQUIRED_DISK_SPACE_SLACK_MULTIPLIER = 1.1;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  /**
   * Checks whether a given path corresponds to a directory.
   *
   * @param path the path to an expected directory.
   * @throws NotDirectoryException if given path is not a directory.
   */
  @Override
  public void checkIsADirectory(Path path) throws NotDirectoryException {
    LOGGER.debug("Validating path [{}] is a directory", path);
    if (!path.toFile().isDirectory()) {
      throw new NotDirectoryException(format("Given path [%s] is not a directory", path));
    }
  }

  /**
   * Checks whether the JVM has writing access to all file trees determined by the given relative paths when resolved against the
   * given parent path.
   *
   * @param parentPath    the parent path which is used for resolving the relative paths.
   * @param relativePaths list of all paths whose underlying tree will be validated for writing access.
   * @throws AccessDeniedException if any path does not have writing accessibility.
   * @throws IOException           if any I/O error occurs.
   */
  @Override
  public void checkWritingAccess(Path parentPath, List<Path> relativePaths) throws IOException {
    LOGGER.debug("Checking writing access for path: {}", parentPath);
    checkWritable(parentPath);
    for (Path relativePath : relativePaths) {
      Path resolvedPath = parentPath.resolve(relativePath);
      checkWritable(resolvedPath);
      try (Stream<Path> fileTree = walk(resolvedPath)) {
        for (Path treePath : fileTree.collect(toList())) {
          checkWritable(treePath);
        }
      }
    }
  }

  private void checkWritable(Path treePath) throws AccessDeniedException {
    if (!isWritable(treePath)) {
      throw new AccessDeniedException(treePath.toString());
    }
  }

  /**
   * Checks whether the JVM has reading access to all file trees determined by the given relative paths when resolved against the
   * given parent path.
   *
   * @param parentPath    the parent path which is used for resolving the relative paths.
   * @param relativePaths list of all paths whose underlying tree will be validated for reading access.
   * @throws AccessDeniedException if any path does not have reading accessibility.
   * @throws IOException           if any I/O error occurs.
   */
  @Override
  public void checkReadingAccess(Path parentPath, List<Path> relativePaths) throws IOException {
    LOGGER.debug("Checking reading access for path: {}", parentPath);
    checkReadable(parentPath);
    for (Path relativePath : relativePaths) {
      Path resolvedPath = parentPath.resolve(relativePath);
      checkReadable(resolvedPath);
      try (Stream<Path> fileTree = walk(resolvedPath)) {
        for (Path treePath : fileTree.collect(toList())) {
          checkReadable(treePath);
        }
      }
    }
  }

  private void checkReadable(Path treePath) throws AccessDeniedException {
    if (!isReadable(treePath)) {
      throw new AccessDeniedException(treePath.toString());
    }
  }

  /**
   * Validates whether there is potentially enough writable space on the file system for the upgrade. The required space is
   * determined either by the space required for the backup creation or the copy of the new files; whoever is the biggest rules.
   * This validation does not guarantee that the process might not fail for lack of space since the conditions on the underlying
   * file system could change in the glimpse of an eye.
   *
   * @param oldMule the directory where the old Mule distro is located.
   * @param newMule the directory where the new Mule distro is located.
   * @param paths   the relative paths of files involved in the upgrade process.
   * @throws {@link IOException} In case of not finding enough usable space.
   */
  @Override
  public void checkEnoughUsableSpace(Path oldMule, Path newMule, List<Path> paths) throws IOException {
    LOGGER.debug("Checking if there's enough usable space for the upgrade from [{}] to [{}]", oldMule, newMule);

    checkExistenceOfFilesInvolvedInTheUpgrade(oldMule, newMule, paths);

    long bytesRequiredToCopyNewFiles = getSumOfBytesTakenByPathsInDistro(newMule, paths);
    long bytesRequiredForBackup = getSumOfBytesTakenByPathsInDistro(oldMule, paths);

    long biggestRequiredSpace =
        bytesRequiredToCopyNewFiles > bytesRequiredForBackup ? bytesRequiredToCopyNewFiles : bytesRequiredForBackup;
    long requiredSpace = (long) (biggestRequiredSpace * REQUIRED_DISK_SPACE_SLACK_MULTIPLIER);
    String requiredSpaceHumanFriendly = byteCountToDisplaySize(requiredSpace);
    LOGGER.debug("The space required for the upgrade is around: {}", requiredSpaceHumanFriendly);

    long usableSpace = getUsableSpace(oldMule);
    String usableSpaceHumanFriendly = byteCountToDisplaySize(usableSpace);
    LOGGER.debug("The usable space: {}", usableSpaceHumanFriendly);

    if (usableSpace <= requiredSpace) {
      throw new IOException(format("Not enough space for the upgrade. Required: %s - Usable: %s", requiredSpace, usableSpace));
    }
  }

  /**
   * Checks whether the underlying OS is Windows.
   *
   * @return true if this is Windows.
   */
  @Override
  public boolean isWindowsOs() {
    return getProperty("os.name").startsWith("Windows");
  }

  /**
   * Recursively copy all the files and directories determined by the given relative paths from the given source directory to the
   * given destination directory. If files to be copied already exist on the destination, then they are deleted prior to its copy.
   *
   * @param srcDir           the directory where the files will be copied from.
   * @param destDir          the directory where the files will be copied to.
   * @param paths            the relative paths that will be resolved against the source dir for copying them to the destination
   *                         directory.
   * @param excludedPathsArg the relative excluded paths that will be resolved against the source dir for avoid copying them to
   *                         the destination directory.
   * @param isDryRunMode     if isDryRunMode is true, the method logs in debug level the files and directories that would be
   *                         copied without actually copying them.
   * @throws IOException if any I/O error occurs.
   */
  @Override
  public void copyFiles(Path srcDir, Path destDir, List<Path> paths, List<Path> excludedPathsArg, boolean isDryRunMode)
      throws IOException {
    List<Path> excludedPaths = new LinkedList<>();
    excludedPathsArg.forEach(excludePath -> excludedPaths.add(srcDir.resolve(excludePath)));

    for (Path path : paths) {
      File source = srcDir.resolve(path).toFile();
      File dest = destDir.resolve(path).toFile();
      if (excludedPaths.contains(source.toPath())) {
        continue;
      }

      if (path.toString().equals("conf")) {
        copyDistroConfFolder(srcDir, path, dest, isDryRunMode);
      } else {
        copyFolderOrFile(source, dest, isDryRunMode);
      }
    }
  }

  void copyDistroConfFolder(Path distroDir, Path path, File dest, boolean isDryRunMode) throws IOException {
    if (!dest.exists()) {
      dest.mkdir();
    }
    File[] files = distroDir.resolve(path).toFile().listFiles();
    for (File confFiles : files) {
      if (!getExtension(confFiles.getName()).equals("lic")) {
        copyFolderOrFile(confFiles, dest.toPath().resolve(confFiles.getName()).toFile(), isDryRunMode);
      }
    }
  }

  void copyFolderOrFile(File source, File dest, boolean isDryRunMode) throws IOException {
    if (source.isDirectory()) {
      copyDir(source, dest, isDryRunMode);
    } else {
      copyFile(source, dest, isDryRunMode);
    }
  }

  void checkExistenceOfFilesInvolvedInTheUpgrade(Path oldMule, Path newMule, List<Path> paths) throws FileNotFoundException {
    checkNoMissingFileInDistro(oldMule, paths);
    checkNoMissingFileInDistro(newMule, paths);
  }

  void checkNoMissingFileInDistro(Path distroHome, List<Path> expectedPathsInDistro) throws FileNotFoundException {
    LOGGER.debug("Checking expected paths [{}] in Mule distro [{}]", expectedPathsInDistro, distroHome);

    List<Path> missingPaths = expectedPathsInDistro.stream()
        .filter(path -> !distroHome.resolve(path).toFile().exists())
        .collect(toList());

    if (!missingPaths.isEmpty()) {
      String joinedMissingPaths = missingPaths.stream()
          .map(path -> distroHome.resolve(path).toString())
          .collect(Collectors.joining(","));
      throw new FileNotFoundException(format("Expected files in distribution were not found: %s", joinedMissingPaths));
    }
  }

  private long getSumOfBytesTakenByPathsInDistro(Path distroHome, List<Path> pathsInvolvedInTheUpgrade) {
    long sumOfSpaceTakenByFiles =
        pathsInvolvedInTheUpgrade.stream().mapToLong(path -> sizeOf(distroHome.resolve(path).toFile())).sum();
    String spaceTakenByFilesHumanFriendly = byteCountToDisplaySize(sumOfSpaceTakenByFiles);
    LOGGER.debug("Paths involved in the upgrade [{}] take: {}", pathsInvolvedInTheUpgrade, spaceTakenByFilesHumanFriendly);
    return sumOfSpaceTakenByFiles;
  }

  long getUsableSpace(Path path) {
    return path.toFile().getUsableSpace();
  }

  private void copyDir(File source, File dest, boolean isDryRunMode) throws IOException {
    if (dest.exists()) {
      LOGGER.debug("Deleting dir [{}] content...", dest);
      if (!isDryRunMode) {
        deleteDirectory(dest);
      }
    }
    LOGGER.debug("Copying dir [{}] content to [{}]...", source, dest);
    if (!isDryRunMode) {
      copyDirectory(source, dest);
    }
  }

  private void copyFile(File source, File dest, boolean isDryRunMode) throws IOException {
    if (dest.exists()) {
      LOGGER.debug("Deleting file [{}]...", dest);
      if (!isDryRunMode) {
        delete(dest.toPath());
      }
    }
    LOGGER.debug("Copying file [{}] to [{}]...", source, dest);
    if (!isDryRunMode) {
      FileUtils.copyFile(source, dest);
    }
  }

  /**
   * Deletes a list of files from a Mule Runtime Distribution.
   * 
   * @param distroPath    Current Mule Runtime Distribution.
   * @param fileNamesList List of files names.
   * @param isDryRunMode  Dry Run Mode flag.
   * @throws IOException if an I/O exception error occurs different that a file not found exception.
   */
  @Override
  public void deleteFilesInsideADistro(Path distroPath, List<String> fileNamesList, boolean isDryRunMode) throws IOException {
    for (String file : fileNamesList) {
      Path filePath = Paths.get(file);
      Path fileToDelete = distroPath.resolve(filePath);

      if (isValidFile(filePath, distroPath) && fileToDelete.toFile().exists()) {
        LOGGER.debug("Deleting file [{}]...", fileToDelete);
        if (!isDryRunMode) {
          delete(fileToDelete);
        }
      }
    }
  }

  /**
   * Copies a list of files from a Mule Runtime Distribution to another.
   *
   * @param sourceMuleDistroPath New Mule Runtime Distribution.
   * @param destMuleDistroPath   Old Mule Runtime Distribution.
   * @param fileNamesList        List of files names.
   * @param isDryRunMode         Dry Run Mode flag.
   * @throws IOException IOException.
   */
  @Override
  public void copyFilesInsideADistro(Path sourceMuleDistroPath, Path destMuleDistroPath, List<String> fileNamesList,
                                     boolean isDryRunMode)
      throws IOException {

    for (String file : fileNamesList) {
      Path filePath = Paths.get(file);
      if (isValidFile(filePath, sourceMuleDistroPath)) {
        Path source = sourceMuleDistroPath.resolve(filePath);
        Path dest = destMuleDistroPath.resolve(filePath);
        LOGGER.debug("Copying file [{}] to [{}]...", source, dest);
        if (!isDryRunMode) {
          FileUtils.copyFile(source.toFile(), dest.toFile());
        }
      }
    }
  }


  /**
   * Checks if the file is valid for any action. During the process, the function validates if the file: - Is included in
   * "{@link FileToBeExcluded}" enum. - Has .lic extension and is inside in the first level of the "conf" directory.
   *
   * @param filePath File path inside the Mule Runtime Distribution.
   * @param srcDir   Path of the Mule Runtime Distribution.
   * @return True if the file is valid, false in any other case.
   */
  boolean isValidFile(Path filePath, Path srcDir) {
    List<Path> excludedPaths = new LinkedList<>();
    String[] filePathArray = filePath.toString().split(Pattern.quote(File.separator));
    int firstDirectoryLevelInsideMuleDistro = 1;

    if (filePathArray.length > firstDirectoryLevelInsideMuleDistro) {
      String fileDir = filePathArray[0];
      String directChildInsideFileDir = filePathArray[1];

      FileToBeExcluded.getAllPaths().forEach(excludePath -> excludedPaths.add(srcDir.resolve(excludePath)));
      File source = srcDir.resolve(filePath).toFile();

      return !(excludedPaths.contains(source.toPath())
          || (fileDir.equals("conf") && getExtension(directChildInsideFileDir).equals("lic")));
    }
    return true;
  }

  /**
   * Deletes the upgrade-tool folder inside a distro if the folder is empty to try to clean the environment after an upgrade or a
   * backup
   *
   * @param muleHome Mule Home path where it is going to try to delete the empty upgrade-tool folder
   * @param dryRun   Dry Run flag, if is in true don't delete the folder and only do the logging
   * @throws IOException IOException.
   */
  @Override
  public void deleteUpgradeFolderIfEmpty(Path muleHome, boolean dryRun) throws IOException {
    Path upgradeToolFolderInDistro = muleHome.resolve(upgradeConfigService.getUpgradeToolFolderName());
    if (!dryRun && Files.exists(upgradeToolFolderInDistro) && FileUtils.isEmptyDirectory(upgradeToolFolderInDistro.toFile())) {
      deleteIfExists(upgradeToolFolderInDistro);
    }
  }

  public boolean areHashesBetweenFilesEqual(Path fileA, Path fileB) throws IOException {
    return getSha256(fileA).equals(getSha256(fileB));
  }

  public boolean areHashesBetweenFilesEqual(byte[] fileContentA, byte[] fileContentB) throws IOException {
    return getSha256(fileContentA).equals(getSha256(fileContentB));
  }

  public String getSha256(Path filePath) throws IOException {
    return getSha256(Files.newInputStream(filePath));
  }

  public String getSha256(InputStream stream) throws IOException {
    return DigestUtils.sha256Hex(stream);
  }

  public String getSha256(byte[] content) {
    return DigestUtils.sha256Hex(content);
  }
}

